##############################################################################################
#  1.13.5
#  purpose: 
#  * convert a dd file into a feature file, so that features can be handled
#    by the T4T-BE in a more generic way. Also fix some dd file inconsistencies
#    to ease the SW structure.
#  * the same procedure is also used for importing production files into T4T-C projects. 
#    In that case the converter gets the EcgDescription element of a PF and converts that into a fea. 
#  
#
#  The structure of the feature file is described in
#  https://jira.int.osram-light.com:6443/display/T4T/Generic+Feature+Model
#
##############################################################################################
#
#   to do:
#   * ext.base uses offset=1: is this different in the parameter API/Schreder?
#   * partial programming: add ALL feature dependencies (only done for DALI/EM/TD)
#   * with the give procedure, no reset to default is possible after import as SetValue is copied to Default
#     In order to fix that the BE must be changed as well....
#
#
#   descoped from v1:
#   * Separation of TD and CF as separate features?
#   * introduce a dummy switch "SSOMode-Enable" for drivers based on OTConfig-1 to lock out SSO for incompatible OM 
#
#
#   done/fixed:
#   * added handling of dynamic luminaire info fields (e.g. date of manufacture) uses hidden properties
#   * added remapping of deprecated LED Module O-15 to O-11 
#   * create td.mo from active operating mode (instead of fixed default) 
#   * fixed OM programming selection for drivers with TDiCorridor-0 SubDevice (fix: lockprogramming=True)
#   * added (ext) module ID handling
#   * fixed d2w -> dw as feature short name
#   * correction of Corridor td.h0: use tdh4 enum type
#   * import of TWA-0 with ledmodule
#   * reject old and too new dd file versions -> particularly important for the import use case
#   * removed "multiple instances" from lum info feature (forced to 1)
#   * copy locations settings from source
#   * convert astro timestamps from source (for all 3 astro-types)
#   * add Unit to property attributes (for use in sumary in FE) for all features. Unit is mandatory, so default is ""
#   * support of new TDCF property TDiCorridor-1:CFPowerOnBehavior   (trail w/ am31194.xml)
#   * switch display name from "DALI Settings" to "Dexal Settings" in case of DX drivers. 
#   * "locked" partial programming attributes for dependent features has 2 cases:
#       a) in same SubDevice (e.g. cur+sso) or
#       b) with joined parameter (e.g. DALI+TD) 
#     concept is implemented: add dep attribute to feature with list of dependent (locked) features
#   * TW support: separation of DALI DT6 and DT8 props 
#     - LED modules and support in OM (via LedModule:Mode)
#   * MultiChannel support added 
#   * import use case: re-map "unsupported" extfade times
#   * consistent handling of DALI fade: joined property for std fade and fast fade, separate for extended fade.
#   * create list of features in OM visible for allowed/active set. Done for: PD,SD,MD,ASTRO,SSO,TDCF.
#       * add feature mappings: touchdim, stepdim, mainsdim, presence, ThermProt,   ...
#       * introduced emum like data types with reference (for properties DALI fade times, TD fade/hold times...),
#         for use in API ("availableValues")
#   * copy setvalue to default to cover import case. Handled as separate function
#     set_to_default(), that can be called from the importer, but used from converter
#   * added information to <Feature> and <Property> elements to enable reverse mapping. Concept:
#       - at Feature level: added Mapping (type) attribute with meaning:
#             0 = direct, no conversion required
#             1..N = mapping follows the hardcoded rule #x
#             999 = undefined yet (e.g. during mapping BE just record a warning)
#       - at Property level: added "Map" (ref) attribute, if this property is mapped 1:1 do dd file
#             if Map is empty, either no mapping is required or specific mapping is implemented in BE
#   * ignore QTi, PTo and 3DIM drivers in T4T-Cloud. How? -> added "Scope" attribute in MetaData
#       if not "C" in scope: ignore the file in this feature file converter
#   * TDCF - mainsfrequency? this shall not be covered in the BE as that's no "programming issue".
#       Instead, the FE shall be able to convert "display values" on user request.
#   * invert the default for cur.en (= not default of ledset.enable), i.e. no direct copy!
#
###########################################################################################

# external python libraries
from lxml import etree as et
from copy import deepcopy
import os
import sys
import time
import ddfile
from  genericFeatureHandler import GenericFeatureHandler

# own libraries
import midnight         # calculation of midnightshift based on location 

debug = 0

# region ############ aux functions and classes ###############################

def resetGlobalVariables():
    global g_maxFeatureInstances
    g_maxFeatureInstances = {}
    global daliDependencies
    daliDependencies = ""
    global subdevice2featureMap
    subdevice2featureMap = {}
    return

def getdriver(structure, n=0):   # get nth driver in file, default is the first driver
                                 # and return it as type "lxml.etree._Element"

    # dd files have <ECGDescription> as their root element. In that case find does not work.
    # Therefore compare the root element.
    # In case of a dd file, the structure itself must be returned and
    # parameter n is ignored as there are no other driver(s) included

    if type(structure) == et._ElementTree:

        if structure.getroot().tag == "ECGDescription":    # i.e. the dd file case
            return structure.getroot()
    else:
        if structure.tag == "ECGDescription":
            return structure 

    devs = structure.findall(".//ECGDescription")

    if not devs:
        raise Exception("No driver data in received file!")

    if (len(devs) > n):     # make sure that nth driver exist, otherwise return None
        return devs[n]

    raise Exception("Less than %d drivers in received file!" % n)


def getsubdevice(structure, name):
    if structure is not None:
        return structure.find(".//SubDevice[@PgmName='%s']" % name)
        # for sub in structure.findall(".//SubDevice"):
        #     if sub.attrib["PgmName"]==name:
        #         return sub


def getproperty(structure, param):
    data = structure.find(".//Property[@PgmName='%s']" % param)
    if data is None:  # None means "not found"
        raise NameError("ValueNotPresent")
    return data

    # currently not used, introduce this in a future refactoring:
    # def getparameter (structure, param, requested_attrib):
    #     data = structure.find("./Property[@PgmName='%s']/Format"%param)
    #     if data == None:
    #         raise NameError("ValueNotPresent")
    #     value = getformatvalue(data, requested_attrib)
    #     return value


def invert_bool(element, name):
    tmp = element.attrib[name]
    if tmp == "0":
        tmp = "1"
    else:
        tmp = "0"
    element.attrib[name] = tmp
    return

def get_bool_value(element, propName: str, default: bool) -> bool:
    """ get boolean value from the property value
        if property does not exist, return default"""
    if element is None:
        return default
    scale = element.find(f"./Property[@PgmName='{propName}']/Format/Scale")
    if scale is None:
        return default
    val = scale.attrib["Default"]
    if val == "0":
        return False
    else:
        return True

def get_display_name(element, propName: str, default: str = "") -> str:
    """ get display name for the property
        if property does not exist, return default"""
    if element is None:
        return default
    display_name = element.find(f"./Property[@PgmName='{propName}']/Display/Name/LangString")
    if display_name is None:
        return default
    return display_name.text

def get_unit(element, propName: str, default: str = "") -> str:
    """ get unit for the property
        if property or unit does not exist, return default"""
    if element is None:
        return default
    unit = element.find(f"./Property[@PgmName='{propName}']/Display/Unit/LangString")
    if unit is None:
        return default
    return unit.text

def create_enum_property(hook, dname, name, map, perm, ref, min, max, default):
    prop = et.SubElement(hook, "Property",
                         DName=dname,
                         Name=name,
                         Map=map,
                         Api="1",
                         Unit="s"
                         )
    if perm is not None:
        prop.attrib["PermW"] = perm

    form = et.SubElement(prop, "Format")
    et.SubElement(form, "EnumValues",
                  Min=min,
                  Max=max,
                  Default=default,
                  Ref=ref
                  )
    return


def copy_format_element(source, instances=0):
                                    # copy the format element according instance parameter 
                                    # "source" must be at <Format> level
                                    # if multiple elements below exist, copy only one and delete instance
                                    # and remove the "Instance" attribute
    if instances == 0:
        print("error, found illegal number of instances")
        return

    elif instances == 1:      # target shall be 1 instance
        sub = source[0]
        new = et.Element("Format")
        tmp = deepcopy(sub)
        tmp.attrib.pop("Instance", None)  # if "Instance" is in sub.attrib: delete it
        new.append(tmp)
        return new

    else:                   # target shall have N instances.
        copied = 0
        new = et.Element("Format")

        for sub in source:
            tmp = deepcopy(sub)
            new.append(tmp)
            copied += 1

        if copied != instances:  # if there are not enough format lines in the source, cry!
            print("error: instance count mismatch: not enough instances in source file!")
            return None

        return new

def createFeaRefEnumPropFromScale(sub, new, propName, feaName, feaDName, unit, min, max, ref, instances):
    details = getproperty(sub, propName)
    prop = et.SubElement(new, "Property",
                            DName=feaDName,
                            Name=feaName,
                            Map=propName,
                            Api="1",
                            Unit=unit
                            )
    form = et.SubElement(prop, "Format")
    if instances == 1:
        value = details.find(".//Format/Scale").attrib["Default"]
        enum = et.SubElement(form, "EnumValues",
                                Min=min,
                                Max=max,
                                Default=value,
                                Ref=ref
                                )
    else:
        for i in range(instances):
            value = details.find(f".//Format/Scale[@Instance='{i}']").attrib["Default"]
            enum = et.SubElement(form, "EnumValues",
                                    Min=min,
                                    Max=max,
                                    Default=value,
                                    Ref=ref,
                                    Instance=str(i)
                                    )
    return

def get_instance_prop(structure, prop, currentInstance, maxinstances=1):
    if maxinstances==1:
        tmp = structure.find("./Property[@PgmName='%s']/Format"%prop)   
        if tmp is not None:                             # avoid exception if prop does not exist
            return tmp[0]                               # return first subelement of what ever type: scale, enumvalue,...
        else:
            return None 
    else:
        elems = structure.find("./Property[@PgmName='%s']/Format"%prop)
        if elems is not None:                            # avoid exception if prop does not exist
            for elem in elems:
                if elem.attrib["Instance"]==str(currentInstance):
                    return elem
            print("** missing instance %d of %s"%(currentInstance, prop))
        else:
            return None

    return None
    

def get_instance_value(structure, currentInstance, maxinstances=1):
    if maxinstances==1:
        tmp = structure.find("./Format")   
        if tmp is not None:                             # avoid exception if prop does not exist
            return tmp[0].attrib["Default"]             # return first subelement of what ever type: scale, enumvalue,...
        else:
            return None 
    else:
        elems = structure.find("./Format")
        if elems is not None:                            # avoid exception if prop does not exist
            for elem in elems:
                if elem.attrib["Instance"]==str(currentInstance):
                    return elem.attrib["Default"]
            print("** missing instance %d of property"%currentInstance)
        else:
            return None

    return None
    
def get_instance_scale_value(structure, currentInstance, maxinstances=1):
    if maxinstances==1:
        tmp = structure.find("./Format")   
        if tmp is not None:                             # avoid exception if prop does not exist
            mul = 1.0
            if "Multiplier" in tmp[0].attrib:
                mul = float(tmp[0].attrib["Multiplier"])
            return int(tmp[0].attrib["Default"]) * mul             # return first subelement of what ever type: scale, enumvalue,...
        else:
            return None 
    else:
        elems = structure.find("./Format")
        if elems is not None:                            # avoid exception if prop does not exist
            for elem in elems:
                if elem.attrib["Instance"]==str(currentInstance):
                    mul = 1.0
                    if "Multiplier" in elem.attrib:
                        mul = float(elem.attrib["Multiplier"])
                    return int(elem.attrib["Default"]) * mul
            print("** missing instance %d of property"%currentInstance)
        else:
            return None

    return None
    
def set_instance_value(structure, value, currentInstance, maxinstances=1):
    if maxinstances==1:
        tmp = structure.find("./Format")   
        if tmp is not None:                             # avoid exception if prop does not exist
            tmp[0].attrib["Default"] = value
        else:
            return
    else:
        elems = structure.find("./Format")
        if elems is not None:                            # avoid exception if prop does not exist
            for elem in elems:
                if elem.attrib["Instance"]==str(currentInstance):
                    elem.attrib["Default"] = value
                    return
            print("** missing instance %d of property"%currentInstance)
        else:
            return
    return

def mirek_to_kelvin(inp: str):
    mirek = int(inp)
    if (mirek == 0):
        mirek = 90
    kelvin = str(round(1000000 / mirek))
    return kelvin


def xy_to_kelvin(x: float, y: float):

    yDif = 0.1858 - y
    if (yDif == 0.0):
        yDif = 0.0000001
    return int((449 * pow(((x - 0.332) / yDif), 3) + 
                3525 * pow(((x - 0.332) / yDif), 2) + 
                6823.3 * ((x - 0.332) / yDif) + 5520.33))
        

def dali_to_level(input: int):
    if input <= 0:
        return 0
    else:
        return round(pow(10, ((input - 1) / (253 / 3) - 1)), ndigits=3)


def omsetting_to_feaname(ref):
    if ref.startswith("AstroA-0") or ref.startswith("Astro-"):   # all versions of astro dim
        return "ast"
    if ref.startswith("D2W-0"):
        return "dw"
    if ref.startswith("GFM-1") or ref.startswith("OperatingMode"):
        return "da"
    if ref.startswith("MD-0"):
        return "md"
    if ref.startswith("PD-0"):
        return "pd"
    if ref.startswith("SD-"):
        return "sd"
    if ref.startswith("TDiCorridor-"):
        return "td"
    if ref.startswith("D2W-0"):
        return "dw"
    if ref.startswith("GFM-0:SSO") or ref.startswith("GFM-1:SSO"):   # GFM-0 is not sufficient...
        return "sso"
    if ref.startswith("MD-0"):
        return "md"
    if ref.startswith("PD-0"):
        return "pd"
    if ref.startswith("SD-"):  # all versions of SD
        return "sd"
    if ref.startswith("td."):   # use case ???
        return "td"
    if ref.startswith("ZeroToTen-0"):
        return "ztt"
    if ref.startswith("O2T-1"): # handled by the generic feature handler
        return None
    print("*** no matching ref found", ref)
    return None

def createFeaProperty(name, dname, map, unit, format=None, maxinstances=1, **kwargs):
    pro = et.Element("Property", Name=name, DName=dname, Map=map, Api="1", Unit=unit)
    form = et.SubElement(pro, "Format")

    if format is None:
        return pro
    for instance in range(maxinstances):
        enum = et.SubElement(form, format)
        for k in kwargs.keys():
            enum.attrib[k] = kwargs[k]

        if maxinstances > 1:
            enum.attrib["Instance"] = str(instance)
    return pro

# some subdevices are missing as copy_LockProgramming is not called (config lock)
subdevice2featureMap = {}

def copy_LockProgramming(source, target):
    if source is not None and target is not None:
        PgmName = source.attrib["PgmName"]
        if PgmName in subdevice2featureMap:
            fname = target.attrib["Name"]
            if fname not in subdevice2featureMap[PgmName]:
                subdevice2featureMap[PgmName] = subdevice2featureMap[PgmName] + "," + fname
        else:
            subdevice2featureMap[PgmName] = target.attrib["Name"]
        if "LockProgramming" in source.attrib:
            target.attrib["LockProgramming"] = source.attrib["LockProgramming"]
        else:
            target.attrib["LockProgramming"] = "false"   # add as default 
        
    # else copy nothing
    return


class astrotimestamp():

    def __init__(self):  # create a timestamp with default = midnight
        self.day=1
        self.hour=0
        self.min=0
    
    def __str__(self):
        return ('{:02d}'.format(self.hour)+":"+'{:02d}'.format(self.min))

    def sub (self, min):
        check = self.day*1440 + self.hour*60 + self.min
        result = check - min
        if result < 0:
            self.day=0
            self.hour=0
            self.min=0
            return self
        elif result >2159: 
            self.day=1
            self.hour=11
            self.min=59
            return self
        else:
            self.day  = int(result / 1440)
            self.hour = int(result % 1440 / 60)
            self.min  = int(result % 60)
            return self

    def add (self, min):
        return self.sub(min * -1)

def ifDriverIsInAstroDT6FixAstroDT8Timings(times):
    # detect if driver is in astro dt6 mode
    if times[7] == 0 and times[8] == 0:
        # set default astro DT6 timings
        defaultTimings = [180, 60, 60, 60, 60, 60, 60, 60, 60]
        for i in range(9):
            times[i] = defaultTimings[i]
    pass

def convert_astromodel(prefix, source, target, isTW, instance=1, maxinstances=1):
    featurePrefix = "ast"
    # get the original dimming schedule settings 
    times = [get_instance_scale_value(source.find(".//Property[@PgmName='%s:DimStartTime']"%prefix), instance, maxinstances)]
    count = 3
    start = 1
    if prefix == "Astro-3":
        count = 8
        start = 2
    if prefix == "Astro-3" and not isTW:
        count = 3
    if prefix == "Astro-3" and isTW:
        featurePrefix = "asttw"
    for i in range(start, start+count):
        times.append(get_instance_scale_value(source.find(f".//Property[@PgmName='{prefix}:DimDuration{i}']"), instance, maxinstances))
    
    if isTW:
        # astro dt8 is sharing ddfile props with the astro dt6 with 5 steps. When importing a driver in astro dt6 mode
        # and than switching to astro dt8 the 10 step values are odd as DimDuration5 .. DimDuration9 are 0
        ifDriverIsInAstroDT6FixAstroDT8Timings(times)

    if prefix == "Astro-2":
        submode = get_instance_value(source.find(".//Property[@PgmName='Astro-2:Control-AstroDimSubMode']"), instance, maxinstances)
    elif prefix == "Astro-3":
        submode = get_instance_value(source.find(".//Property[@PgmName='Astro-3:Control-AstroDimSubMode']"), instance, maxinstances)
    elif prefix == "Astro-1":
        submode = get_instance_value(source.find(".//Property[@PgmName='Astro-1:Control-TimingRef']"), instance, maxinstances)
    else:
        submode = "0" # some driver models have no SubMode, then use astro-based mode

    lon = get_instance_scale_value(source.find(".//Property[@PgmName='%s:Longitude']"%prefix), instance, maxinstances)
    lat = get_instance_scale_value(source.find(".//Property[@PgmName='%s:Latitude']"%prefix), instance, maxinstances)
    ts = get_instance_scale_value(source.find(".//Property[@PgmName='%s:UTCTimeShift']"%prefix), instance, maxinstances)

    timestamp = astrotimestamp()    # start from midnight

    if submode == "0":              # for astro-based mode
        if (lat!=0 or lon !=0 or ts!=0):    # and only for non-default locations
            mid = midnight.CalculateMidnightShift(lat, lon, ts)
            timestamp.add(mid)              # include midnightshift            
        timestamp.sub(times[0])
    elif submode =="1":             # for time-based mode
        timestamp.sub(1440)             # create 00:00 as startingpoint
        timestamp.add(times[0])               # and add t0 to it
    else:                           # for SD-triggered mode
        timestamp.sub(1440)             # but dont add t0 (ignore it)

    times[0] = str(timestamp)
    astro_time_props = { "%s:DimStartTime"%prefix: (f"{featurePrefix}.t0", "Time 2", True, times[0]) }
    for i in range (0, count):
        timestamp.add(times[i+1])
        times[i+1] = str(timestamp)
        ddPropName = f"{prefix}:DimDuration{start+i}"
        feaPropName = f"{featurePrefix}.t{i+1}"
        astro_time_props[ddPropName] = (feaPropName, f"Time {i+3}", True, times[i+1])

    # create the astrodim time properties
    for elem in astro_time_props:
        feaProp = target.find(f".//Property[@Name='{astro_time_props[elem][0]}']/Format")
        if feaProp is None:
            details = getproperty(source, elem)
            en = et.SubElement(target, "Property",
                                DName=astro_time_props[elem][1],
                                Name=astro_time_props[elem][0],
                                Map="",
                                Api="1" if astro_time_props[elem][2] else "0",
                                Unit="hh:mm")
            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]
            feaProp = et.SubElement(en, "Format")
        time = et.SubElement(feaProp, "TimeStamp", Default=astro_time_props[elem][3])
        if maxinstances > 1:
            time.attrib["Instance"] = str(instance)
    return

# single instance props just return the value, multi instance props as '|' seperated string
# e.g. 1|0
def getMultiInstancePropertyValues(structure, propName):
    pformat = structure.find(f".//Property[@PgmName='{propName}']/Format")
    valueCount = len(pformat)
    values = [""] * valueCount
    for i in range(valueCount):
        if valueCount == 1:
            values[i] = pformat[i].attrib["Default"]
        else:
            instance = pformat[i].attrib["Instance"]
            values[int(instance)] = pformat[i].attrib["Default"]
    multiValueStr = '|'.join(values)
    # print(propName, valueCount, multiValueStr)
    return multiValueStr

def isSettingDisabled(setting, ref):
    if setting.attrib["Ref"] != ref:
        return False
    val = setting.attrib["Value"]
    if val == "0" or val == "0|0" or val == "0|0|0" or val == "0|0|0|0":
        return True
    return False

# endregion
# region ############ feature related functions ###############################
g_maxFeatureInstances = {}
def getMaxFeatureInstanceCount(featureName):
    if featureName in g_maxFeatureInstances:
        return g_maxFeatureInstances[featureName]
    return 1

def getFeatureInstanceCount(subdevice, feature, constants):
    instances = 1
    feaName = feature.attrib["Name"]
    if "MultipleInstance" in subdevice.attrib and subdevice.attrib["MultipleInstance"] == "true":
        tmp = constants.find(".//Constant[@Name='MaxInstances']").attrib["Value"]
        instances = int(tmp) 
        # in multi-instance drivers there are two DT6 instances required per one DT8
        if feaName == "d8":
            instances = int(instances / 2)
    feature.attrib["Instances"] = str(instances)
    global g_maxFeatureInstances
    g_maxFeatureInstances[feaName] = instances
    return instances

def fill_metadata(structure, metadata=None, constants=None, filename=""):

    # handle all elements individually, if some needs to be deleted?
    # metadata_elements = {}
    # data = dict()
    # data["gtin"] = structure.find(".//MetaData/GTIN").text
    # data["fw"]   = structure.find(".//MetaData/FW_Major").text
    # data["hw"]   = structure.find(".//MetaData/HW_Major").text
    # data["name"] = structure.find(".//MetaData/DeviceTypeName").text
    # data["bc"]   = structure.find(".//MetaData/BasicCode").text

    meta_ignore = ["Slowprogramming", "ECGType", "DaliName", "MonitoringDataVersion", "Vendor", "Application", 
                   "DALIClearAddress"]

    # goto MetaData elements in source
    md = structure.find(".//MetaData")

    # devide them into constants and "real" meta data elements
    for elem in md:
        name = elem.tag
        if (name == "RatedDimmingRange"):
            # convert to 2 constants
            tmp = et.Element("Constant", Name="RatedDimmingRangeMax", Value=elem.attrib["Max"])
            constants.append(tmp)
            tmp = et.Element("Constant", Name="RatedDimmingRangeMin", Value=elem.attrib["Min"])
            constants.append(tmp)
        elif (name == "LEDSetVersion"):
            tmp = et.Element("Constant", Name=name, Value=elem.text)
            constants.append(tmp)
        elif (name == "InterfaceType"):    # add NFC capability to constants, so that mappings can take this into account
            it = elem.text
            if it.startswith("NFC"):
                if it == "NFC":
                    tmp = et.Element("Constant", Name="NfcType", Value="0")
                else:
                    tmp = et.Element("Constant", Name="NfcType", Value=it[-1:])  # only last digit
                constants.append(tmp)
        elif (name == "MaxInstance"):
            tmp = et.Element("Constant", Name="MaxInstances", Value=elem.text)
            constants.append(tmp)
        elif name in meta_ignore:
            continue  # ignore, do nothing, go to next elem
        else:
            metadata.append(deepcopy(elem))
    return

def getPartialOemKeyLockedFeatures(ddfileXml):
    if getsubdevice(ddfileXml, "Pwd1-1") is None:
        return None
    subs = ddfileXml.findall(".//SubDevice")
    lockedFeatures = []
    for subdevice in subs:
        name = subdevice.attrib["PgmName"]
        if name == "Pwd1-1":
            break
        if (name in subdevice2featureMap):
            lockedFeatures.append(subdevice2featureMap[name])
    return ",".join(lockedFeatures)

def fill_operatingmodes(genericFeatureHandler, structure, om=None, features=None, feanames=None, filename=""):

    # static part for operating mode handling
    td0_opsmode = """
        <OperatingModes>
            <Mode DName="DALI and TD" Name="dali_td" >
                <Setting Ref="TDiCorridor-0:TDConfig-TDCFEnable" Value="1"/>
                <Setting Ref="TDiCorridor-0:TDControl-TDOperate" Value="0"/>
                <Setting Ref="TDiCorridor-0:TDControl-CFOperate" Value="0"/>
            </Mode>
            <Mode DName="DALI only" Name="dali">
                <Setting Ref="TDiCorridor-0:TDConfig-TDCFEnable" Value="0"/>
                <Setting Ref="TDiCorridor-0:TDControl-TDOperate" Value="0"/>
                <Setting Ref="TDiCorridor-0:TDControl-CFOperate" Value="0"/>
            </Mode>
            <Mode DName="TD" Name="td">
                <Setting Ref="TDiCorridor-0:TDConfig-TDCFEnable" Value="1"/>
                <Setting Ref="TDiCorridor-0:TDControl-TDOperate" Value="1"/>
                <Setting Ref="TDiCorridor-0:TDControl-CFOperate" Value="0"/>
            </Mode>
            <Mode DName="CF" Name="cf">
                <Setting Ref="TDiCorridor-0:TDConfig-TDCFEnable" Value="1"/>
                <Setting Ref="TDiCorridor-0:TDControl-TDOperate" Value="1"/>
                <Setting Ref="TDiCorridor-0:TDControl-CFOperate" Value="1"/>
            </Mode>
        </OperatingModes>"""
    parser = et.XMLParser(remove_blank_text=True)
    td0_opsmode_element = et.XML(td0_opsmode, parser)

    all_opsmode_names = {
        "0-10 v": "0to10",
        "0-10v/pwm/resistor": "0to10",
        "1* dali dt6 & soft switch off (1 channel)": "dali_sso",
        "1* dali dt6 & soft switch off (2 channels, synched)": "dali_sso_dual",
        "1* dali dt6 & soft switch off (2 synched outputs)": "dali_sso_dual",
        "1* dali dt6 (1 channel)": "dali",
        "dali dt6": "dali",
        "1* dali dt6 (1 ch independently)": "dali",
        "1* dali dt6 (2 channels, synched)": "dali_dual",
        "1* dali dt6 (2 synched outputs)": "dali_dual",
        "1* dali dt6 (4 synched outputs)": "dali_4",
        "1* dali dt6 (dim to warm output)": "dali_d2w",
        "1* dali dt8 (tunable white output)": "dali8tw",
        "1* dali dt8 (tunable white)": "dali8tw",
        "2* dali dt6 & soft switch off (2 channels, independently controllable)": "2dali_sso",
        "2* dali dt6 & soft switch off (2 independent outputs)": "2dali_sso",
        "2* dali dt6 (2 channels, independently controllable)": "2dali",
        "2* dali dt6 (2 independent outputs)": "2dali",
        "3* dali dt6 (3 independent outputs)": "3dali",
        "4* dali dt6 (4 independent outputs)": "4dali",
        "astrodim": "astro",
        "astrodim (dali)": "astro",
        "astrodim + presence detection": "astro_pd",
        "astrodim + presence detection (dali)": "astro_pd",
        "astrodim/lsi/dali (wiring selection)": "astro_sd",
        "autodetect, astrodim-lsi inverse (dali)": "astro_sdi",
        "autodetect, astrodim-stepdim inverse (dali)": "astro_sdi",
        "autodetect: dali or touchdim": "dali_td", 
        "autodetect, dali-2 & touchdim": "dali_td",      # import case
        "autodetect: dali dt6 or touchdim": "dali_td",        
        "autodetect, on/off-lsi inverse (dali)": "onoff_sdi",
        "autodetect, on/off-stepdim inverse (dali)": "onoff_sdi",
        "autodetect (reset to factory-default)": "auto",
        "corridor functionality": "cf",        
        "corridor": "cf",                           # import case only
        "cf": "cf",                                 # import case only
        "corridor functionality (1 channel)": "cf",
        "corridor functionality (2 channels, synchronized output)": "cf_dual",
        "corridor functionality (2 synched outputs)": "cf_dual",
        "corridor functionality (2ch sync)": "cf_dual",
        "corridor functionality (4 synched outputs)": "cf_4",
        "dali": "dali",
        "dali only": "dali",
        "bluetooth mesh": "dali",
        "dali & soft switch off": "dali_sso",
        "dali (data), soft switch off": "dali_sso",
        "dali (data)": "data",
        "dali and enhanced options": "dali_sso",
        "dali & enhanced options": "dali_sso",      # import case only
        "dali and td": "dali_td",                   # import case only
        "dali-2 & soft switch off": "dali_sso",     # import case only
        "dali-2 and soft switch off": "dali_sso",   # import case only
        "dali-2": "dali",                           # import case only
        "dali-2 and td": "dali_td",                 # import case only
        "dali-2 only": "dali",                      # import case only
        "lsi (dali)": "sd",
        "lsi inverse (dali)": "sdi",
        "mainsdim (dali)": "md",
        "no dimming (on-off)": "onoff",
        "on/off": "onoff",
        "on/off (2 independent outputs)": "2onoff",
        "stepdim (dali)": "sd",
        "stepdim inverse (dali)": "sdi",
        "stepdim/astrodim/dali (wiring selection)": "astro_sd",
        "td": "td",                                 # import case only
        "touchdim": "td",
        "touchdim (1 channel)": "td",
        "touchdim (2 channels, synchronized output)": "td_dual",
        "touchdim (2ch sync)": "td_dual",
        "touchdim (2 independent outputs)": "td_2but",
        "touchdim (2 synched outputs)": "td_dual",
        "touchdim (4 synched outputs)": "td_4",
        "touchdim (dim to warm output)": "td_d2w",
        "touchdim (dim to warm)": "td_d2w",
        "touchdim (tunable white output)": "td_tw",
        "touchdim (tunable white)": "td_tw",
        "tunable white - dali dt8": "dali8tw",
        "bilevel/midnight/dali (wiring selection)": "astro_sd",
        "midnight (dali)": "astro",
        "midnight": "astro",
        "midnight + presence detection (dali)": "astro_pd",
        "bilevel (dali)": "sd",
        "bilevel lsi (dali)": "sd",
        "bilevel inverse (dali)": "sdi",
        "bilevel lsi inverse (dali)": "sdi",
        "autodetect, on/off-bilevel inverse (dali)": "onoff_sdi",
        "autodetect, on/off-bilevel lsi inverse (dali)": "onoff_sdi",
        "autodetect,on/off-bilevel inverse (dali)": "onoff_sdi",
        "autodetect, midnight-bilevel inverse (dali)": "astro_sdi",
        "autodetect, midnight-bilevel lsi inverse (dali)": "astro_sdi",
        "amp dim (dali)": "md",
        "default": "default", }

    # prepare a dict with all features in the driver
    features_in_driver = {}
    for elem in feanames:
        features_in_driver[elem] = 1  # by default a feature is there just once

    # convert into list of all features in this driver (e.g. "da,cur,clm,em,ops,td,inf") 
    features_in_driver_string = ",".join(map(str, sorted(features_in_driver)))

    check_ops_modes = {}
    om_dependent_features = {}

    modes = structure.findall(".//Mode")
    for mode in modes:      # iterate over all modes in dd file

        features_in_this_mode = deepcopy(features_in_driver)    # start with the complete list of features in the driver
        # and then remove those not active in this Operating Mode

        if "Name" in mode.attrib:   # only handle modes with a name. Other modes are "spacer entries", hence skip them here
            orig_mode = mode.attrib["Name"].strip().lower()
            new_mode = et.SubElement(om, "Mode", DName=mode.attrib["Name"])

            # get fea opmode name
            opModeName = None
            if "FeaName" in mode.attrib:
                opModeName = mode.attrib["FeaName"].strip().lower()
            else:
                cleanOpModeName = " ".join(orig_mode.split()) # remove whitespace duplicates between words
                if cleanOpModeName in all_opsmode_names:
                    opModeName = all_opsmode_names[cleanOpModeName]
            if opModeName is not None:
                new_mode.set("Name", opModeName)

                # check if Opsmodes in Name attribute are unique per driver
                if opModeName in check_ops_modes:
                    print("Error: OPS Mode names in driver %s are not unique: " % filename, end="")
                    for elem in check_ops_modes:
                        print(elem, end=" ")
                    print("")
                else:
                    check_ops_modes[opModeName] = True

            else:
                new_mode.set("Name", "missing")
                print("Error: opsmode (" + orig_mode + ") not covered by conversion in file ", filename)

            # copy all settings for this operating mode and remove the disabled features (if possible)
            settings = mode.findall("./Setting")
            keep_td = False

            # remove features based on opmode names
            spd_opmodes = ["astro", "astro_tw", "astro_dual", "2astro"]
            for fea in features_in_driver:
                if fea == "pod" and "dali" in opModeName:
                    del features_in_this_mode["pod"]
                    continue
                if fea == "spd" and opModeName not in spd_opmodes:
                    del features_in_this_mode["spd"]
                    continue

            for setting in settings:
                new_mode.append(deepcopy(setting))
                tmp = omsetting_to_feaname(setting.attrib["Ref"])
                if tmp is not None:
                    om_dependent_features[tmp] = 1

                # if one of the conditions applies, the (sub)feature has to be removed from the "visibility list"
                for fea in features_in_driver:
                    if fea == "sso" and ((setting.attrib["Ref"] == "GFM-0:SSOMode-Allowed" and setting.attrib["Value"] == "0") or
                                         (setting.attrib["Ref"] == "GFM-1:SSOMode-Allowed" and setting.attrib["Value"] == "0")):
                        del features_in_this_mode["sso"]
                        continue
                    if fea == "ast" and (isSettingDisabled(setting, "Astro-1:Control-AstroEnable") or
                                         isSettingDisabled(setting, "Astro-2:Control-AstroEnable") or
                                         isSettingDisabled(setting, "Astro-3:Control-AstroEnable")):
                        del features_in_this_mode["ast"]
                        if "asttw" in features_in_this_mode:
                            del features_in_this_mode["asttw"]
                        continue
                    if fea == "sd" and isSettingDisabled(setting, "SD-0:Control-SDEnable"):
                        del features_in_this_mode["sd"]
                        continue
                    if fea == "sd" and isSettingDisabled(setting, "SD-1:Control-SDEnable"):
                        del features_in_this_mode["sd"]
                        if "sd8" in features_in_this_mode:
                            del features_in_this_mode["sd8"]
                        continue
                    if fea == "pd" and isSettingDisabled(setting, "PD-0:Control-PDEnable"):
                        del features_in_this_mode["pd"]
                        if "pd8" in features_in_this_mode:
                            del features_in_this_mode["pd8"]
                        continue
                    if fea == "md" and setting.attrib["Ref"] == "MD-0:Control-MDEnable" and setting.attrib["Value"] == "0":
                        del features_in_this_mode["md"]
                        continue
                    if fea == "ztt" and setting.attrib["Ref"] == "ZeroToTen-0:Enable" and setting.attrib["Value"] == "0":
                        del features_in_this_mode["ztt"]
                        continue
                    if fea == "ast" and setting.attrib["Ref"] == "AstroA-0:Enable" and setting.attrib["Value"] == "0":
                        del features_in_this_mode["ast"]
                        continue
                    if fea == "dw" and setting.attrib["Ref"] == "D2W-0:Enable" and setting.attrib["Value"] == "0":
                        del features_in_this_mode["dw"]
                        continue
                    if fea == "dgps" and ((setting.attrib["Ref"] == "OperatingMode" and setting.attrib["Value"] != "140")):
                        del features_in_this_mode["dgps"]
                        continue

                    # for "td" it is the other way round: if one pre-condition matches, keep it!
                    # note: this needs more work, if we split TD and CF features later (= potential feature request)
                    if fea == "td" and ((setting.attrib["Ref"] == "OperatingMode" and setting.attrib["Value"] == "129") or
                                        (setting.attrib["Ref"] == "TDiCorridor-1:TDConfig-TDCFEnable" and setting.attrib["Value"] == "1") or
                                        (setting.attrib["Ref"] == "TDiCorridor-5:TDConfig-TDCFEnable" and setting.attrib["Value"] == "1")):
                        keep_td = True

                    # finally use genericFeatureHandler
                    if (genericFeatureHandler.featureNotAvailableForOpMode(fea, opModeName, setting.attrib["Ref"], setting.attrib["Value"])):
                        if fea in features_in_this_mode:
                            del features_in_this_mode[fea]
                        continue


            # finally, if no keep_td condition was found, remove the "td" feature, as it is not active
            if not keep_td and "td" in features_in_this_mode:
                del features_in_this_mode["td"]

            # modify the featurelist according instances 

            # 1) find out the multiplicity of the multi-instance features and set it in dict
            tmp = mode.find("./Module[@Ref='DaliSet:Mode']")
            if tmp is not None:
                val = int(tmp.attrib["Value"])
                # defininion of Daliset:Mode parameter
                # 0: Remove DALI and TW related features
                # 1: 1 channel simple
                # 2: 1 channel with TW
                # 3: 2 channels without TW
                # 4: RGBWAF
                if val == 0:
                    features_in_this_mode["da"] = 0
                    if "dadr" in features_in_this_mode:
                        features_in_this_mode["dadr"] = 0
                    features_in_this_mode["d8"] = 0
                    features_in_this_mode["rgbw"] = 0
                elif val == 1:
                    features_in_this_mode["da"] = 1
                    if "dadr" in features_in_this_mode:
                        features_in_this_mode["dadr"] = 1
                    features_in_this_mode["d8"] = 0
                    features_in_this_mode["rgbw"] = 0
                elif val == 2:
                    features_in_this_mode["da"] = 1
                    if "dadr" in features_in_this_mode:
                        features_in_this_mode["dadr"] = 1
                    features_in_this_mode["d8"] = 1
                    features_in_this_mode["rgbw"] = 0
                elif val == 3:
                    features_in_this_mode["da"] = 2
                    if "dadr" in features_in_this_mode:
                        features_in_this_mode["dadr"] = 2
                    features_in_this_mode["d8"] = 0
                    features_in_this_mode["rgbw"] = 0
                elif val == 4:
                    features_in_this_mode["da"] = 1
                    if "dadr" in features_in_this_mode:
                        features_in_this_mode["dadr"] = 1
                    features_in_this_mode["d8"] = 0
                    features_in_this_mode["rgbw"] = 1

            tmp = mode.find("./Module[@Ref='OpsCur2:Mode']")
            if tmp is not None:
                val = int(tmp.attrib["Value"])
                # defininion of OpsCur2:Mode parameter
                # 0: 2x current via LED Module settings
                # 1: 2x current (with "joined limit" validation)
                # 2: N/A
                # 3: 1x current
                # 4: 2x independend channels including enabled

                # ignore this parameter !!!
                # if val == 2:
                #     features_in_this_mode["cur"] = val
                if val == 4:
                    features_in_this_mode["cur"] = 2

            tmp = mode.find("./Module[@Ref='TDCF:Mode']")     # this is indicator only present in multi-channel drivers
            if tmp is not None:
                val = int(tmp.attrib["Value"])
                # defininion of TDCF:Mode parameter
                # 0: no TDCF not present
                # 1: 1 ch "standard" TDCF
                # 2: 2 ch 
                # 3: 1 ch with TW
                if val == 0:
                    features_in_this_mode["td"] = 0
                else:
                    features_in_this_mode["td"] = 1
                if val == 3:
                    features_in_this_mode["d8"] = 1

                # to let the API user know which TD parameters are valid, is transported via the td.mo property 
                tmp = et.Element("Setting", Ref="td.mo", Value=str(val))
                new_mode.append(tmp)

            tmp = mode.find("./Module[@Ref='Global:Instances']")
            if tmp is not None:
                val = int(tmp.attrib["Value"])
                if val > 1:
                    if "em" in features_in_this_mode:
                        features_in_this_mode["em"] = val
                    if "clm" in features_in_this_mode:
                        features_in_this_mode["clm"] = val
                    if "ops" in features_in_this_mode:
                        features_in_this_mode["ops"] = val
                    if "ast" in features_in_this_mode:
                        features_in_this_mode["ast"] = val
                    if "sd" in features_in_this_mode:
                        features_in_this_mode["sd"] = val
                    if "pd" in features_in_this_mode:
                        features_in_this_mode["pd"] = val

            tmp = mode.find("./Module[@Ref='TF:Instances']")
            if tmp is not None:
                val = int(tmp.attrib["Value"])
                if val > 1:
                    features_in_this_mode["tf"] = val

            tmp = mode.find("./Module[@Ref='LedModule:Mode']")
            if tmp is not None:
                val = int(tmp.attrib["Value"])
                if val == 0:
                    features_in_this_mode["led"] = 0
                    features_in_this_mode["d8"] = 0
                    if "asttw" in features_in_this_mode:
                        del features_in_this_mode["asttw"]
                    if "sd8" in features_in_this_mode:
                        del features_in_this_mode["sd8"]
                    if "pd8" in features_in_this_mode:
                        del features_in_this_mode["pd8"]
                else:
                    features_in_this_mode["led"] = 1
                    if "ast" in features_in_this_mode:
                        del features_in_this_mode["ast"]

            # convert the resulting "feature dict" into list and add the instance counter
            om_fea = ""
            for fea in features_in_this_mode.keys():
                if features_in_this_mode[fea] != 0:    # include a feature only if it is active in this mode
                    maxFeature = getMaxFeatureInstanceCount(fea)
                    instances = min(maxFeature, features_in_this_mode[fea])
                    om_fea += fea + ':' + str(instances) + ','
            if om_fea != "":
                om_fea = om_fea[:-1]  # remove the last comma from list

            new_mode.set("Features", om_fea)
    genOpModeDeps = genericFeatureHandler.getOpModeDependencies(feanames)
    om_dependent_features.update(genOpModeDeps)
    # finally add the collected operating mode dependent features to the new Operating Mode Feature Element
    om_string = ",".join(map(str, sorted(om_dependent_features)))
    
    om_in_source = structure.find(".//OperatingModes")  # get the required setting 
    if om_in_source is not None:
        om_programming = om_in_source.attrib["Programming"]
        om_lockprogramming = om_in_source.get("LockProgramming", "false")
    else:                                       # import case: defaults for old file with no OM
        om_programming = "true"                 # thus programming is possible but
        om_lockprogramming = "true"             # must not be disabled (as it is in T4T-D)

    fea = et.SubElement(features, 
                        "Feature", 
                        DName = "Operating Mode", 
                        Name = "om", 
                        Dep = om_string, 
                        Programming = om_programming,   
                        LockProgramming = om_lockprogramming,
                        Instances = "1",
                        Mapping = "0")      

    if len(check_ops_modes) == 0: 
        # len=0 means: no ops mode yet! As the API requires at least one OM, add it here.
        # This is required for drivers with the old TD feature.
        # or other import cases
        # if no OM can be added, raise an error!

        td0 = structure.find(".//SubDevice[@PgmName='TDiCorridor-0']")     
            # drivers with this subdevice have no opsmodes,
            # but it can be added from template
        if td0 is not None:
            modes = td0_opsmode_element.findall(".//Mode")
            for mode in modes:
                tmp = deepcopy(mode)
                tmp.attrib["Features"] = features_in_driver_string
                om.append(tmp)
            fea.attrib["Dep"] = "da,td"
        else:
            print("Warning: no opsmode found in driver %s; adding a default" % filename)  # import case: old files may have no OM.
            # fix: add a "default om"
            # agreement with BE: drivers w/ 1 OM need no Settings in the OM element
            mode = et.SubElement(om, "Mode",
                                 Name="default",
                                 DName="Default",
                                 Features=features_in_driver_string
                                )
    else:   # opsmode section exists and has at least one entry -> ok
        if debug > 2:
            print("Info: driver has the following opsmodes: ", check_ops_modes)
        
        ### find the active operating mode 
        oms = structure.findall(".//OperatingModes/Mode")      # scan all defined OMs
        active_opsmode = None
        for om in oms:
            if om.get("Name", None) != None:    # special case: no Name = empty OM => skip that
                om_is_active = True 
                props = om.findall ("./Setting")
                # print(om.attrib["Name"], len(props))
                for prop in props:
                    value_in_driver = getMultiInstancePropertyValues(structure, prop.attrib["Ref"])
                    value_in_omsection = prop.attrib["Value"]
                    if '|' not in value_in_omsection and '|' in value_in_driver:
                        values = value_in_driver.split('|')
                        value_in_driver = values[0]

                    if value_in_driver != value_in_omsection :      # found a non-matching setting 
                        om_is_active = False                        # => this is not the active OM
                        break                                       # stop processing this OM, go to the next one
                    
                if om_is_active == True:    # if all Properties match and om_is_active is still true, then the active one has been found
                    active_opsmode = om     # active OM found

                    # warning: in case of multiple "active OpModes" (dd file error!), this will take the last one                 

        if active_opsmode == None:                # if search was not successful, no continuation possible
            print("error in input file: no active OM") 
        else:
            if debug>2: print("** active OM is ", active_opsmode.attrib["Name"])

            tdmode = active_opsmode.find(".//Module[@Ref='TDCF:Mode']")
            if tdmode != None:
                tdmo = features.find(".//Feature/Property[@Name='td.mo']/Format/Scale")
                tdmo.attrib["Default"]= tdmode.attrib["Value"]
    return


def getsettings_0t010(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="ztt", DName="0-10 V", Mapping="0", Instances="1")   # no extra mapping is required. handled by the multiplier

    sub = getsubdevice(structure, "ZeroToTen-0")   # "Zero to 10V" feature based on ZeroToTen-0
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)                              # same for LockProgramming

        details = getproperty(sub, "ZeroToTen-0:MinDimmingLevel")
        prop = et.SubElement(new, "Property",
                             DName="Minimum Dimming Level",
                             Name="ztt.mi",
                             Map="ZeroToTen-0:MinDimmingLevel",
                             Api="1",
                             Unit="%")

        source = details.find("./Format/Scale")
        # min = int(source.attrib["Min"])
        max = int(source.attrib["Max"])
        # default = int(source.attrib["Default"])
        # min_adapted = round(min/max*100)
        # default_adapted = round(default/max*100)
        # multiplier = max/100

        form = et.SubElement(prop, "Format")
        scale = et.SubElement(form, "Scale")

        scale.attrib["Min"] = source.attrib["Min"]
        scale.attrib["Max"] = source.attrib["Max"]
        scale.attrib["Default"] = source.attrib["Default"]
        scale.attrib["Multiplier"] = str(100.0 / max)

        if "PermW" in details.attrib:
            prop.attrib["PermW"] = details.attrib["PermW"]

        ddfile.createFeaPropertyFromDDProperty(new, sub, "ZeroToTen-0:Dim2Off", "ztt.off")

        # get/add the switches of ZeroToTen-0 feature
        details = getproperty(sub, "ZeroToTen-0:Enable")
        tmp = details.find("./Format/Scale")
        if tmp is not None:
            val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="ZeroToTen-0:Enable", Map="ZeroToTen-0:Enable", SetValue=val)

        # get/add the constants of ZeroToTen-0 feature
        ztt_const_props = {
            "ZeroToTen-0:MaxDimmingVoltage": ["ZeroToTen_MaxDimmingVoltage", ""],
            "ZeroToTen-0:MinDimmingVoltage": ["ZeroToTen_MinDimmingVoltage", ""],
            "ZeroToTen-0:MaxDimmingLevel": ["ZeroToTen_MaxDimmingLevel", ""],
        }

        for elem in ztt_const_props:
            details = getproperty(sub, elem)
            tmp = details.find("./Format/Scale")
            if tmp is not None:
                val = tmp.attrib["Default"]
                ztt_const_props[elem][1] = val
            et.SubElement(constants,
                          "Constant",
                          Name=ztt_const_props[elem][0],
                          Value=val)
        
        voltageProp = createFeaProperty("ztt.v", "", "", "", "Scale", 1,
                          Default=ztt_const_props["ZeroToTen-0:MinDimmingVoltage"][1], 
                          Min=ztt_const_props["ZeroToTen-0:MinDimmingVoltage"][1],
                          Max=ztt_const_props["ZeroToTen-0:MaxDimmingVoltage"][1])
        new.append(voltageProp)

        return new

    if debug > 1:
        print("Info: no zero to 10 feature in the driver.")
    return None


def getsettings_astro(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="ast", DName="AstroDIM", Instances="1")

    sub = getsubdevice(structure, "Astro-1")   # astrodim feature based on Astro-1
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)                              # same for LockProgramming

        new.attrib["Mapping"] = "21"

        astro1_scale_props = {
            "Astro-1:NominalLevel": ("ast.l1", "DIM Level 1", True, "%"),
            "Astro-1:DimLevel1": ("ast.l2", "DIM Level 2", True, "%"),
            "Astro-1:DimLevel2": ("ast.l3", "DIM Level 3", True, "%"),
            "Astro-1:DimLevel3": ("ast.l4", "DIM Level 4", True, "%"),
            "Astro-1:DimLevel4": ("ast.l5", "DIM Level 5", True, "%"),
            "Astro-1:Latitude": ("ast.la", "Latitude", False, "°"),
            "Astro-1:Longitude": ("ast.lo", "Longitude", False, "°"),
            "Astro-1:UTCTimeShift": ("ast.ts", "Time Shift", False, "h"),
        }

        # copy add all astro-1 scale properties
        for elem in astro1_scale_props:

            details = getproperty(sub, elem)
            en = et.SubElement(new, "Property",
                               DName=astro1_scale_props[elem][1],
                               Name=astro1_scale_props[elem][0],
                               Map=elem,
                               Api="1" if astro1_scale_props[elem][2] else "0",
                               Unit=astro1_scale_props[elem][3])
            en.append(copy_format_element(details.find("./Format"), 1))
            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

        # create astro-1 timestamp properties
        ###... tbd:
        #     if "PermW" in details.attrib:
        #        en.attrib["PermW"] = details.attrib["PermW"]

        convert_astromodel("Astro-1", sub, new, False)

        # handle the fade times, using default settings 
        astfs = sub.find(".//Property[@PgmName='Astro-1:StartupFadeTime']/Format/Scale").attrib["Default"]
        astfd = sub.find(".//Property[@PgmName='Astro-1:AstroDIMFadeTime']/Format/Scale").attrib["Default"]
        astfe = sub.find(".//Property[@PgmName='Astro-1:SwitchOFFFadeTime']/Format/Scale").attrib["Default"]

        # 1) Startup Fade:
        details = getproperty(sub, "Astro-1:StartupFadeTime")
        prop = et.SubElement(new, "Property",
                             DName="Startup Fadetime",
                             Name="ast.fs",
                             Map="Astro-1:StartupFadeTime",
                             Api="1",
                             Unit="s"
                             )
        form = et.SubElement(prop, "Format")
        enum = et.SubElement(form, "EnumValues",
                             Min="0",
                             Max="240",
                             Default=astfs,
                             Ref="astroswitchfade"
                             )
        # 2) Dim Fade:
        details = getproperty(sub, "Astro-1:AstroDIMFadeTime")
        prop = et.SubElement(new, "Property",
                             DName="AstroDIM Fade Time",
                             Name="ast.fd",
                             Map="Astro-1:AstroDIMFadeTime",
                             Api="1",
                             Unit="s"
                             )
        form = et.SubElement(prop, "Format")
        enum = et.SubElement(form, "EnumValues",
                             Min="0",
                             Max="240",
                             Default=astfd,
                             Ref="astrodimfade"
                             )
        # 3) Switch Off Fade:
        details = getproperty(sub, "Astro-1:SwitchOFFFadeTime")
        prop = et.SubElement(new, "Property",
                             DName="Switch Off Fade Time",
                             Name="ast.fe",
                             Map="Astro-1:SwitchOFFFadeTime",
                             Api="1",
                             Unit="s"
                             )
        form = et.SubElement(prop, "Format")
        enum = et.SubElement(form, "EnumValues",
                             Min="0",
                             Max="255",
                             Default=astfe,
                             Ref="astroswitchfade"
                             )

        # handle submode of Astro-1 (has only time-based)
        details = getproperty(sub, "Astro-1:Control-TimingRef")
        tmp = details.find("./Format/Scale")        # there can be a scale or enumvalue in this property
        if tmp is not None:
            val = tmp.attrib["Default"]
        else:
            tmp = details.find("./Format/EnumValues")
            val = tmp.attrib["Default"]

        prop = et.SubElement(new, "Property",
                             Name="ast.mo",
                             DName="Submode",
                             Map="Astro-1:Control-TimingRef",
                             Api="1",
                             Unit="")
        form = et.SubElement(prop, "Format")
        scale = et.SubElement(form, "Scale",
                              Min="0",
                              Max="1",
                              Default=val
                              )

        # switch(es) for Astro-1 feature
        details = getproperty(sub, "Astro-1:Control-AstroEnable")
        tmp = details.find("./Format/Scale")
        if tmp is not None:
            val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="Astro-1:Control-AstroEnable", Map="Astro-1:Control-AstroEnable", SetValue=val)

    if sub is None:
        sub = getsubdevice(structure, "Astro-2")   # astrodim feature based on Astro-2
        if sub is not None:
            instances = getFeatureInstanceCount(sub, new, constants)

            new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
            copy_LockProgramming(sub, new)                              # same for LockProgramming

            new.attrib["Mapping"] = "22"

            astro2_scale_props = {
                "Astro-2:NominalLevel": ("ast.l1", "Output Level 1", True, "%"),
                "Astro-2:DimLevel1": ("ast.l2", "Output Level 2", True, "%"),
                "Astro-2:DimLevel2": ("ast.l3", "Output Level 3", True, "%"),
                "Astro-2:DimLevel3": ("ast.l4", "Output Level 4", True, "%"),
                "Astro-2:DimLevel4": ("ast.l5", "Output Level 5", True, "%"),
                "Astro-2:Latitude": ("ast.la", "Latitude", False, "°"),
                "Astro-2:Longitude": ("ast.lo", "Longitude", False, "°"),
                "Astro-2:UTCTimeShift": ("ast.ts", "Time Shift", False, "h"), }

            # copy add all astrodim-2 <scale> properties
            for elem in astro2_scale_props:

                details = getproperty(sub, elem)
                en = et.SubElement(new, "Property",
                                   DName=astro2_scale_props[elem][1],
                                   Name=astro2_scale_props[elem][0],
                                   Map=elem,
                                   Api="1" if astro2_scale_props[elem][2] else "0",
                                   Unit=astro2_scale_props[elem][3]
                                   )
                en.append(copy_format_element(details.find("./Format"), instances))
                if "PermW" in details.attrib:
                    en.attrib["PermW"] = details.attrib["PermW"]

            # create reference schedule
            for i in range(instances):
                convert_astromodel("Astro-2", sub, new, False, i, instances)


            # handle the fade times: use default settings            
            createFeaRefEnumPropFromScale(sub, new, "Astro-2:StartupFadeTime", "ast.fs", "Startup Fadetime", "s", "0", "3600", "astroswitchfade16b", instances)
            createFeaRefEnumPropFromScale(sub, new, "Astro-2:AstroDIMFadeTime", "ast.fd", "AstroDIM Fade Time", "s", "0", "240", "astrodimfade", instances)
            createFeaRefEnumPropFromScale(sub, new, "Astro-2:SwitchOFFFadeTime", "ast.fe", "Switch Off Fade Time", "s", "0", "255", "astroswitchfade", instances)

            details = getproperty(sub, "Astro-2:Control-AstroDimSubMode")
            tmp = details.find("./Format/Scale")        # can be there a scale or enumvalue in this property
            if tmp is not None:
                default = tmp.attrib["Default"]
                min = tmp.attrib["Min"]
                max = tmp.attrib["Max"]
            else:
                tmp = details.find("./Format/EnumValues")
                default = tmp.attrib["Default"]
                if "Min" in tmp.attrib:                 # import case: missing min/max in EnumValues --> dig out the list and find the actual min/max
                    min = tmp.attrib["Min"]
                else:
                    min = 255
                    for elem in details.findall("./Format/EnumValues/EnumValue"):
                        if int(elem.attrib["Value"]) < min:
                            min = int(elem.attrib["Value"])
                    if debug > 1:
                        print("Info: missing min in AstroSubMode; using determined min")
                    min = str(min)
                if "Max" in tmp.attrib:
                    max = tmp.attrib["Max"]
                else:
                    max = 0
                    for elem in details.findall("./Format/EnumValues/EnumValue"):
                        if int(elem.attrib["Value"]) > max:
                            max = int(elem.attrib["Value"])
                    if debug > 1:
                        print("Info: missing max in AstroSubMode; using determined max")
                    max = str(max)

            prop = et.SubElement(new, "Property",
                                 Name="ast.mo",
                                 DName="Submode",
                                 Map="Astro-2:Control-AstroDimSubMode",
                                 Api="1", 
                                 Unit="")
            form = et.SubElement(prop, "Format")
            if (instances == 1):
                scale = et.SubElement(form, "Scale",
                                    Min=min,
                                    Max=max,
                                    Default=default
                                    )
            else:
                for i in range(instances):
                    value = get_instance_value(details, i, instances)
                    enum = et.SubElement(form, "Scale",
                                            Min=min,
                                            Max=max,
                                            Default=value,
                                            Instance=str(i)
                                            )

            # switch(es) for Astro-2 feature
            val = getMultiInstancePropertyValues(sub, "Astro-2:Control-AstroEnable")
            et.SubElement(switches, "Switch", Name="Astro-2:Control-AstroEnable", Map="Astro-2:Control-AstroEnable", SetValue=val)

    if sub is None:
        sub = getsubdevice(structure, "Astro-3")   # astrodim feature based on Astro-3
        if sub is not None:
            instances = getFeatureInstanceCount(sub, new, constants)

            new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
            copy_LockProgramming(sub, new)                              # same for LockProgramming

            new.attrib["Mapping"] = "35"

            astro2_scale_props = {
                "Astro-3:NominalLevel": ("ast.l1", "Output Level 1", True, "%"),
                "Astro-3:DimLevel2": ("ast.l2", "Output Level 2", True, "%"),
                "Astro-3:DimLevel3": ("ast.l3", "Output Level 3", True, "%"),
                "Astro-3:DimLevel4": ("ast.l4", "Output Level 4", True, "%"),
                "Astro-3:DimLevel10": ("ast.l5", "Output Level 5", True, "%"),
                "Astro-3:Latitude": ("ast.la", "Latitude", False, "°"),
                "Astro-3:Longitude": ("ast.lo", "Longitude", False, "°"),
                "Astro-3:UTCTimeShift": ("ast.ts", "Time Shift", False, "h"), }

            # copy add all astrodim-2 <scale> properties
            for elem in astro2_scale_props:

                details = getproperty(sub, elem)
                en = et.SubElement(new, "Property",
                                   DName=astro2_scale_props[elem][1],
                                   Name=astro2_scale_props[elem][0],
                                   Map=elem,
                                   Api="1" if astro2_scale_props[elem][2] else "0",
                                   Unit=astro2_scale_props[elem][3]
                                   )
                en.append(copy_format_element(details.find("./Format"), instances))
                if "PermW" in details.attrib:
                    en.attrib["PermW"] = details.attrib["PermW"]

            # create reference schedule
            for i in range(instances):
                convert_astromodel("Astro-3", sub, new, False, i, instances)


            # handle the fade times: use default settings            
            createFeaRefEnumPropFromScale(sub, new, "Astro-3:StartupFadeTime", "ast.fs", "Startup Fadetime", "s", "0", "3600", "astroswitchfade16b", instances)
            createFeaRefEnumPropFromScale(sub, new, "Astro-3:AstroDIMFadeTime", "ast.fd", "AstroDIM Fade Time", "s", "0", "240", "astrodimfade", instances)
            createFeaRefEnumPropFromScale(sub, new, "Astro-3:SwitchOFFFadeTime", "ast.fe", "Switch Off Fade Time", "s", "0", "255", "astroswitchfade", instances)

            details = getproperty(sub, "Astro-3:Control-AstroDimSubMode")
            tmp = details.find("./Format/Scale")        # can be there a scale or enumvalue in this property
            if tmp is not None:
                default = tmp.attrib["Default"]
                min = tmp.attrib["Min"]
                max = tmp.attrib["Max"]
            else:
                tmp = details.find("./Format/EnumValues")
                default = tmp.attrib["Default"]
                if "Min" in tmp.attrib:                 # import case: missing min/max in EnumValues --> dig out the list and find the actual min/max
                    min = tmp.attrib["Min"]
                else:
                    min = 255
                    for elem in details.findall("./Format/EnumValues/EnumValue"):
                        if int(elem.attrib["Value"]) < min:
                            min = int(elem.attrib["Value"])
                    if debug > 1:
                        print("Info: missing min in AstroSubMode; using determined min")
                    min = str(min)
                if "Max" in tmp.attrib:
                    max = tmp.attrib["Max"]
                else:
                    max = 0
                    for elem in details.findall("./Format/EnumValues/EnumValue"):
                        if int(elem.attrib["Value"]) > max:
                            max = int(elem.attrib["Value"])
                    if debug > 1:
                        print("Info: missing max in AstroSubMode; using determined max")
                    max = str(max)

            prop = et.SubElement(new, "Property",
                                 Name="ast.mo",
                                 DName="Submode",
                                 Map="Astro-3:Control-AstroDimSubMode",
                                 Api="1", 
                                 Unit="")
            form = et.SubElement(prop, "Format")
            if (instances == 1):
                scale = et.SubElement(form, "Scale",
                                    Min=min,
                                    Max=max,
                                    Default=default
                                    )
            else:
                for i in range(instances):
                    value = get_instance_value(details, i, instances)
                    enum = et.SubElement(form, "Scale",
                                            Min=min,
                                            Max=max,
                                            Default=value,
                                            Instance=str(i)
                                            )

            # switch(es) for Astro-3 feature
            val = getMultiInstancePropertyValues(sub, "Astro-3:Control-AstroEnable")
            et.SubElement(switches, "Switch", Name="Astro-3:Control-AstroEnable", Map="Astro-3:Control-AstroEnable", SetValue=val)

    if sub is None:
        sub = getsubdevice(structure, "AstroA-0")   # astrodim feature based on AstroA-0 SubDevice (Nafta)
        if sub is not None:
            new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
            copy_LockProgramming(sub, new)                              # same for LockProgramming

            new.attrib["Mapping"] = "23"

            astroa_scale_props = {
                "AstroA-0:NominalLevel": ("ast.l1", "Output Level 1", True, "%"),
                "AstroA-0:DimLevel1": ("ast.l2", "Output Level 2", True, "%"),
                "AstroA-0:DimLevel2": ("ast.l3", "Output Level 3", True, "%"),
                "AstroA-0:DimLevel3": ("ast.l4", "Output Level 4", True, "%"),
                "AstroA-0:DimLevel4": ("ast.l5", "Output Level 5", True, "%"),
                "AstroA-0:Latitude": ("ast.la", "Latitude", False, "°"),
                "AstroA-0:Longitude": ("ast.lo", "Longitude", False, "°"),
                "AstroA-0:UTCTimeShift": ("ast.ts", "Time Shift", False, "h"),
            }

            # astroa_time_props = {
            #     "AstroA-0:DimStartTime": ("ast.t0", "Time 2", True, "22:00"),
            #     "AstroA-0:DimDuration1": ("ast.t1", "Time 3", True, "00:00"),
            #     "AstroA-0:DimDuration2": ("ast.t2", "Time 4", True, "02:00"),
            #     "AstroA-0:DimDuration3": ("ast.t3", "Time 5", True, "04:00"),
            # }

            # copy add all astrodim-a <scale> properties
            for elem in astroa_scale_props:

                details = getproperty(sub, elem)
                en = et.SubElement(new, "Property",
                                   DName=astroa_scale_props[elem][1],
                                   Name=astroa_scale_props[elem][0],
                                   Map=elem,
                                   Api="1" if astroa_scale_props[elem][2] else "0",
                                   Unit=astroa_scale_props[elem][3])
                en.append(copy_format_element(details.find("./Format"), 1))
                if "PermW" in details.attrib:
                    en.attrib["PermW"] = details.attrib["PermW"]

            # handle reference schedule
            convert_astromodel("AstroA-0", sub, new, False)

            # handle the fade times: use fixed settings
            astfs = sub.find(".//Property[@PgmName='AstroA-0:StartupFadeTime']/Format/Scale").attrib["Default"]
            astfd = sub.find(".//Property[@PgmName='AstroA-0:AstroDIMFadeTime']/Format/Scale").attrib["Default"]
            astfe = sub.find(".//Property[@PgmName='AstroA-0:SwitchOFFFadeTime']/Format/Scale").attrib["Default"]

            # 1) Startup Fade:
            details = getproperty(sub, "AstroA-0:StartupFadeTime")
            prop = et.SubElement(new, "Property",
                                 DName="Startup Fadetime",
                                 Name="ast.fs",
                                 Map="AstroA-0:StartupFadeTime",
                                 Api="1",
                                 Unit="s"
                                 )
            form = et.SubElement(prop, "Format")
            enums = et.SubElement(form, "EnumValues",
                                  Min="0",
                                  Max="0",
                                  Default=astfs,
                                  )
            enum = et.SubElement(enums, "EnumValue", Label="0", Value="0")
            # 2) Dim Fade:
            details = getproperty(sub, "AstroA-0:AstroDIMFadeTime")
            prop = et.SubElement(new, "Property",
                                 DName="AstroDIM Fade Time",
                                 Name="ast.fd",
                                 Map="AstroA-0:AstroDIMFadeTime",
                                 Api="1",
                                 Unit="s"
                                 )
            form = et.SubElement(prop, "Format")
            enum = et.SubElement(form, "EnumValues",
                                 Min="0",
                                 Max="240",
                                 Default=astfd,
                                 Ref="astrodimfade"
                                 )
            # 3) Switch Off Fade:
            details = getproperty(sub, "AstroA-0:SwitchOFFFadeTime")
            prop = et.SubElement(new, "Property",
                                 DName="Switch Off Fade Time",
                                 Name="ast.fe",
                                 Map="AstroA-0:SwitchOFFFadeTime",
                                 Api="1",
                                 Unit="s"
                                 )
            form = et.SubElement(prop, "Format")
            enums = et.SubElement(form, "EnumValues",
                                  Min="255",
                                  Max="255",
                                  Default="255",
                                  )
            et.SubElement(enums, "EnumValue", Label="OFF", Value="255")

            prop = et.SubElement(new, "Property",
                                 Name="ast.mo",
                                 DName="Submode",
                                 Map="",
                                 Api="1",
                                 Unit="")
            form = et.SubElement(prop, "Format")
            et.SubElement(form, "Scale",
                           Min="0",
                           Max="0",
                           Default="0"
                           )

            # get the switch(es) for AstroA-0 feature
            details = getproperty(sub, "AstroA-0:Enable")
            tmp = details.find("./Format/Scale")
            if tmp is not None:
                val = tmp.attrib["Default"]
            et.SubElement(switches, "Switch", Name="AstroA-0:Enable", Map="AstroA-0:Enable", SetValue=val)

    # common procedures for all astrodim subdevice types
    if sub is not None:
        # copy the HwResources
        hw_resouces = sub.findall("./HwResource")
        for hwr in hw_resouces:
            new.append(deepcopy(hwr))

        # copy the constants --> there are no constants in AstroDim

        return new

    if debug > 1:
        print("Info: no astrodim feature in the driver.")
    return None

def getsettings_astrotw(structure, switches=None, constants=None, filename=""):
    instances = 1
    new = et.Element("Feature", Name="asttw", DName="AstroDIM TW", Instances=str(instances))
    new.attrib["Mapping"] = "34"

    sub = getsubdevice(structure, "Astro-3")   # astrodim feature based on Astro-3
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)                              # same for LockProgramming

        astro3_scale_props = {
            "Astro-3:NominalLevel": ("asttw.l1", "Output Level 1", True, "%"),
            "Astro-3:DimLevel2": ("asttw.l2", "Output Level 2", True, "%"),
            "Astro-3:DimLevel3": ("asttw.l3", "Output Level 3", True, "%"),
            "Astro-3:DimLevel4": ("asttw.l4", "Output Level 4", True, "%"),
            "Astro-3:DimLevel5": ("asttw.l5", "Output Level 5", True, "%"),
            "Astro-3:DimLevel6": ("asttw.l6", "Output Level 6", True, "%"),
            "Astro-3:DimLevel7": ("asttw.l7", "Output Level 7", True, "%"),
            "Astro-3:DimLevel8": ("asttw.l8", "Output Level 8", True, "%"),
            "Astro-3:DimLevel9": ("asttw.l9", "Output Level 9", True, "%"),
            "Astro-3:DimLevel10": ("asttw.l10", "Output Level 10", True, "%"),
            "Astro-3:Latitude": ("asttw.la", "Latitude", False, "°"),
            "Astro-3:Longitude": ("asttw.lo", "Longitude", False, "°"),
            "Astro-3:UTCTimeShift": ("asttw.ts", "Time Shift", False, "h"), }

        # copy add all astrodim-2 <scale> properties
        for elem in astro3_scale_props:

            details = getproperty(sub, elem)
            en = et.SubElement(new, "Property",
                                DName=astro3_scale_props[elem][1],
                                Name=astro3_scale_props[elem][0],
                                Map=elem,
                                Api="1" if astro3_scale_props[elem][2] else "0",
                                Unit=astro3_scale_props[elem][3]
                                )
            en.append(copy_format_element(details.find("./Format"), instances))
            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

        # create reference schedule
        for i in range(instances):
            convert_astromodel("Astro-3", sub, new, True)

        colorProps = {                 # list of dali dt8 parameters to be handled automatically and in the same way
            "Astro-3:Colour_1": ("asttw.col1", ""),
            "Astro-3:Colour_2": ("asttw.col2", ""),
            "Astro-3:Colour_3": ("asttw.col3", ""),
            "Astro-3:Colour_4": ("asttw.col4", ""),
            "Astro-3:Colour_5": ("asttw.col5", ""),
            "Astro-3:Colour_6": ("asttw.col6", ""),
            "Astro-3:Colour_7": ("asttw.col7", ""),
            "Astro-3:Colour_8": ("asttw.col8", ""),
            "Astro-3:Colour_9": ("asttw.col9", ""),
            "Astro-3:Colour_10": ("asttw.col10", ""),
            "Astro-3:Colour_OnOff": ("asttw.ci", "")}

        createColorProperties(structure, ".//SubDevice[@PgmName='Astro-3']", colorProps, new)

        # handle the fade times: use default settings            
        createFeaRefEnumPropFromScale(sub, new, "Astro-3:StartupFadeTime", "asttw.fs", "Startup Fadetime", "s", "0", "3600", "astroswitchfade16b", instances)
        createFeaRefEnumPropFromScale(sub, new, "Astro-3:AstroDIMFadeTime", "asttw.fd", "AstroDIM Fade Time", "s", "0", "240", "astrodimfade", instances)
        createFeaRefEnumPropFromScale(sub, new, "Astro-3:SwitchOFFFadeTime", "asttw.fe", "Switch Off Fade Time", "s", "0", "255", "astroswitchfade", instances)

        details = getproperty(sub, "Astro-3:Control-AstroDimSubMode")
        tmp = details.find("./Format/Scale")        # can be there a scale or enumvalue in this property
        if tmp is not None:
            default = tmp.attrib["Default"]
            min = tmp.attrib["Min"]
            max = tmp.attrib["Max"]
        else:
            tmp = details.find("./Format/EnumValues")
            default = tmp.attrib["Default"]
            if "Min" in tmp.attrib:                 # import case: missing min/max in EnumValues --> dig out the list and find the actual min/max
                min = tmp.attrib["Min"]
            else:
                min = 255
                for elem in details.findall("./Format/EnumValues/EnumValue"):
                    if int(elem.attrib["Value"]) < min:
                        min = int(elem.attrib["Value"])
                if debug > 1:
                    print("Info: missing min in AstroSubMode; using determined min")
                min = str(min)
            if "Max" in tmp.attrib:
                max = tmp.attrib["Max"]
            else:
                max = 0
                for elem in details.findall("./Format/EnumValues/EnumValue"):
                    if int(elem.attrib["Value"]) > max:
                        max = int(elem.attrib["Value"])
                if debug > 1:
                    print("Info: missing max in AstroSubMode; using determined max")
                max = str(max)

        prop = et.SubElement(new, "Property",
                                Name="asttw.mo",
                                DName="Submode",
                                Map="Astro-3:Control-AstroDimSubMode",
                                Api="1", 
                                Unit="")
        form = et.SubElement(prop, "Format")
        scale = et.SubElement(form, "Scale",
                            Min=min,
                            Max=max,
                            Default=default
                            )
        # common procedures for all astrodim subdevice types
        return new

    if debug > 1:
        print("Info: no astrodim TW feature in the driver.")
    return None

def getsettings_sdtw(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="sd8", DName="", Instances="1")
    new.attrib["Mapping"] = "32"

    colorProps = {                 # list of dali dt8 parameters to be handled automatically and in the same way
        "SD-1:Colour_1": ("sd8.col1", ""),
        "SD-1:Colour_2": ("sd8.col2", ""),
        "SD-1:Colour_3": ("sd8.col3", "")
    }

    createColorProperties(structure, ".//SubDevice[@PgmName='SD-1']", colorProps, new)

    # common procedures for all astrodim subdevice types
    if len(new) == 3:
        return new

    if debug > 1:
        print("Info: no sd TW feature in the driver.")
    return None

def getsettings_pdtw(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="pd8", DName="", Instances="1")
    new.attrib["Mapping"] = "32"

    colorProps = {                 # list of dali dt8 parameters to be handled automatically and in the same way
        "PD-0:Colour": ("pd8.col", "")
    }

    createColorProperties(structure, ".//SubDevice[@PgmName='PD-0']", colorProps, new)

    # common procedures for all astrodim subdevice types
    if len(new) == 1:
        return new

    if debug > 1:
        print("Info: no pd TW feature in the driver.")
    return None

def fixClmEndOfTable(fea, instances):
    if instances == 1:
        for index in range(1, 9):
            time = fea.find(".//Property[@Name='clm.t%s']/Format/Scale"%index).attrib["Default"]
            if time == "255":
                fea.find(".//Property[@Name='clm.l%s']/Format/Scale"%index).attrib["Default"] = '255'
    else:
        for instance in range(instances):
            for index in range(1, 9):
                time = fea.find(f".//Property[@Name='clm.t{index}']/Format/Scale[@Instance='{instance}']").attrib["Default"]
                if time == "255":
                    fea.find(f".//Property[@Name='clm.l{index}']/Format/Scale[@Instance='{instance}']").attrib["Default"] = '255'
    return

def getsettings_clm(structure, switches=None, constants=None, filename=""):
    new = et.Element("Feature", Name="clm", DName="Constant Lumen", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "ConstLum-0")
    if sub is not None:

        instances = getFeatureInstanceCount(sub, new, constants)

        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from souce
        copy_LockProgramming(sub, new)                              # same for LockProgramming


        it = constants.find(".//Constant[@Name='NfcType']")
        if it is not None:
            nfcversion = it.attrib["Value"]
        else:
            nfcversion = "0"

        if nfcversion == "3":
            new.attrib["Mapping"] = "8"    # in this case a differnt mapping has to be applied (adaptation to fixed x-times)
            nfc3 = True
        else:
            new.attrib["Mapping"] = "0"     # all other cases
            nfc3 = False

        details = getproperty(sub, "ConstLum-0:Command-ClmEnable")
        # if nfc3 is True:
        #     tmp = et.SubElement(new, "Property", DName="Enable", Name="clm.en", Map="", Api="1", Unit="")
        # else:
        tmp = et.SubElement(new, "Property", DName="Enable", Name="clm.en", Map="ConstLum-0:Command-ClmEnable", Api="1", Unit="")

        tmp.append(copy_format_element(details.find("./Format"), instances))

        # check how many "clm curve points" are in the driver
        points = 8
        for index in range(points):
            try:
                details = getproperty(sub, "ConstLum-0:Time" + str(index + 1))
            except NameError:
                if index == 4:
                    points = 4
                    new.attrib["Mapping"] = "7"
                else:
                    print("Error in dd file - CLM: less than 8 points, but not 4")
                break

        lasttime = lastlevel = None

        for index in range(points):
            details = getproperty(sub, "ConstLum-0:Time" + str(index + 1))
            tmp = et.SubElement(new, "Property", DName="Operating Time %s" % str(index + 1), 
                                Name="clm.t%s" % str(index + 1), Api="1", Unit="kh")

            if index == 0:
                # if nfc3 is True:
                #     tmp.attrib["Map"] = ""  # BE to determine mapping
                # else:
                #     tmp.attrib["Map"] = "ConstLum-0:Time1"  # BE to use fixed mapping
                tmp.attrib["Map"] = "ConstLum-0:Time1"  

            else:
                #if points == 4 or nfc3 is True:
                if points == 4:
                    tmp.attrib["Map"] = ""  # no direct mapping possible
                else:
                    tmp.attrib["Map"] = "ConstLum-0:Time%s" % str(index + 1)  # BE to use fixed mapping

            lasttime = details.find("./Format")
            tmp.append(copy_format_element(lasttime, instances))
            if "PermW" in details.attrib:
                tmp.attrib["PermW"] = details.attrib["PermW"]

            details = getproperty(sub, "ConstLum-0:AdjustmentLevel" + str(index + 1))
            tmp = et.SubElement(new, "Property",
                                DName="Output Level %s" % str(index + 1),
                                Name="clm.l%s" % str(index + 1),
                                Api="1",
                                Unit="%")

            if index == 0:
                # if nfc3 == True:
                #     tmp.attrib["Map"] = ""  # BE to determine mapping
                # else:
                #     tmp.attrib["Map"] = "ConstLum-0:AdjustmentLevel1"  # BE to use fixed mapping
                tmp.attrib["Map"] = "ConstLum-0:AdjustmentLevel1"  
            else:
                # if points == 4 or nfc3 is True:
                if points == 4:
                    tmp.attrib["Map"] = ""  # no direct mapping possible
                else:
                    tmp.attrib["Map"] = "ConstLum-0:AdjustmentLevel%s" % str(index + 1)  # BE to use fixed mapping

            lastlevel = details.find("./Format")
            elem = lastlevel.find("./Scale")            # prevent clm levels greater than 100. "error" in 4 drivers
            if int(elem.attrib["Max"])>100:
                elem.attrib["Max"]='100'

            tmp.append(copy_format_element(lastlevel, instances))
            if "PermW" in details.attrib:
                tmp.attrib["PermW"] = details.attrib["PermW"]

        if points == 4:  # fill up the last 4 elements
            for index in range(4, 8):
                # times must be strong monotone increasing up to off
                time_property = lasttime.find(".//Scale")
                time = int(time_property.attrib["Default"])
                if time < 255:
                    time += 1
                    time_property.attrib["Default"] = str(time)

                tmp = et.SubElement(new, "Property",
                                    DName="Time T%s" % str(index + 1),
                                    Name="clm.t%s" % str(index + 1),
                                    Map="",
                                    Api="1",
                                    Unit="kh")
                tmp.append(copy_format_element(lasttime, instances))
                if "PermW" in details.attrib:
                    tmp.attrib["PermW"] = details.attrib["PermW"]
                tmp = et.SubElement(new, "Property",
                                    DName="Adjustmentlevel L%s" % str(index + 1),
                                    Name="clm.l%s" % str(index + 1),
                                    Map="",
                                    Api="1",
                                    Unit="%")
                tmp.append(copy_format_element(lastlevel, instances))
                if "PermW" in details.attrib:
                    tmp.attrib["PermW"] = details.attrib["PermW"]

        fixClmEndOfTable(new, instances)
        return new

    sub = getsubdevice(structure, "ConstLumA-0")
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from souce
        copy_LockProgramming(sub, new)                              # same for LockProgramming

        details = getproperty(sub, "ConstLumA-0:Enable")
        tmp = et.SubElement(new, "Property", DName="Enable", Name="clm.en", Map="ConstLumA-0:Enable", Api="1", Unit="")
        tmp.append(copy_format_element(details.find("./Format"), 1))

        # ConstLumA-0 driver always have 8 points

        for index in range(8):
            details = getproperty(sub, "ConstLumA-0:Time" + str(index + 1))
            tmp = et.SubElement(new, "Property",
                                DName="Time T%s" % str(index + 1),
                                Name="clm.t%s" % str(index + 1),
                                Map="ConstLumA-0:Time" + str(index + 1),
                                Api="1",
                                Unit="kh")
            tmp.append(copy_format_element(details.find("./Format"), 1))
            if "PermW" in details.attrib:
                tmp.attrib["PermW"] = details.attrib["PermW"]

            details = getproperty(sub, "ConstLumA-0:AdjustmentLevel" + str(index + 1))
            tmp = et.SubElement(new, "Property", DName="Adjustmentlevel L%s" % str(index + 1),
                                Name="clm.l%s" % str(index + 1),
                                Map="ConstLumA-0:AdjustmentLevel%s" % str(index + 1),
                                Api="1",
                                Unit="%")

            tmp.append(copy_format_element(details.find("./Format"), 1))
            if "PermW" in details.attrib:
                tmp.attrib["PermW"] = details.attrib["PermW"]

        fixClmEndOfTable(new, 1)
        return new

    if debug > 1:
        print("Info: No CLM feature in the driver.")
    return None

def detectraw_import(structure):
    if "IsRaw" in structure.find(".//SubDevices").attrib:
        return True
    return False

def getScalePropertyValue(sub, subName, propname):
    name = f"{subName}:{propname}"
    scale = sub.find(f".//Property[@PgmName='{name}']/Format/Scale")
    return scale, scale.attrib["Default"]

# returns masterkey, isEncrypted and the key.allvalue
def getMasterKeyMapping12(subDevicePwd1, subDevicePwd2):
    scale = subDevicePwd1.find(".//Property[@PgmName='Pwd1-1:EncryptedPassword']/Format/Scale")
    if scale is not None:
        masterKey = scale.attrib["Default"]
        isEncrypted = "1"
        scale = subDevicePwd2.find(".//Property[@PgmName='Pwd2-1:EncryptedPassword']/Format/Scale")
        if scale is not None and masterKey != "0" and scale.attrib["Default"] == "0":
            return masterKey, isEncrypted, "0"
        return masterKey, isEncrypted, "1"
    scale = subDevicePwd1.find(".//Property[@PgmName='Pwd1-1:Password']/Format/Scale")
    if scale is not None:
        masterKey = scale.attrib["Default"]
        isEncrypted = "1"
        scale = subDevicePwd2.find(".//Property[@PgmName='Pwd2-1:Password']/Format/Scale")
        if scale is not None and masterKey != "0" and scale.attrib["Default"] == "0":
            return masterKey, isEncrypted, "0"
        return masterKey, isEncrypted, "1"
    return "0", "0", "1"

def isKeyEncrypted(isRaw, key):
    if isRaw and key != "0":
        return "1"
    return "0"

def getsettings_configlock(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="key", DName="Configuration Lock", Instances="1")
    map_type = 0
    enabledValue = "0"
    permudef = "4294967295"
    permsdef = "4294967295"
    masterKey = "0"
    serviceKey = "0"
    isEncrypted = "0"
    keyAll = "1"

    isRaw = detectraw_import(structure)
    subName = "MSK-0"
    sub = getsubdevice(structure, subName)   # Configuration Lock feature based on MSK-0 subdevice, e.g. in AM28411
    if sub is not None:
        map_type = 13
        programming = sub.attrib["Programming"]
        lockprogramming = sub.get("LockProgramming", "false")
        configlock_scale_props = [
            ("key.en", "Enable", "", True),
            ("key.mp", "Master Key", "MSK-0:NewMasterPwd", True),
            ("key.sp", "Service Key", "MSK-0:ServicePwd", True),
            ("key.pu", "Permission User", "MSK-0:PermUser", True),
            ("key.ps", "Permission Service", "MSK-0:PermService", True),
            ("key.em", "Encrypted Master Key", "", True),
            ("key.es", "Encrypted Service Key", "", True)]

        if isRaw:
            bio = getsubdevice(structure, "BIO-0")
            if bio is not None:
                subName = "BIO-0"
                sub = bio
            _, masterKey = getScalePropertyValue(sub, subName, 'MasterPwd')
            _, serviceKey = getScalePropertyValue(sub, subName, 'ServicePwd')
        permu, permudef = getScalePropertyValue(sub, subName, 'PermUser')
        perms, permsdef = getScalePropertyValue(sub, subName, 'PermService')

    else:
        sub = getsubdevice(structure, "PwmConfig-0")   # Configuration Lock feature based on PwmConfig-0 subdevice, e.g. in AM10971.xml

        if sub is not None:
            map_type = 14
            programming = sub.attrib["Programming"]
            lockprogramming = sub.get("LockProgramming", "false")
            configlock_scale_props = [
                ("key.en", "Enable", "", True),
                ("key.mp", "Master Password", "PwmConfig-0:MasterPwd", True),
                ("key.sp", "Service Password", "PwmConfig-0:ServicePwd", True),
                ("key.pu", "Permission User", "PwmConfig-0:PermUser", True),
                ("key.ps", "Permission Service", "PwmConfig-0:PermService", True),
                ("key.em", "Encrypted Master Key", "", True),
                ("key.es", "Encrypted Service Key", "", True)]

            permu = sub.find(".//Property[@PgmName='PwmConfig-0:PermUser']/Format/Scale")
            permudef = permu.attrib["Default"]
            perms = sub.find(".//Property[@PgmName='PwmConfig-0:PermService']/Format/Scale")
            permsdef = perms.attrib["Default"]
            if isRaw:
                scale = sub.find(".//Property[@PgmName='PwmConfig-0:MasterPwd']/Format/Scale")
                masterKey = scale.attrib["Default"]
                scale = sub.find(".//Property[@PgmName='PwmConfig-0:ServicePwd']/Format/Scale")
                serviceKey = scale.attrib["Default"]

            # get and add the dependent features as Dep attribute
            dependent = dict()
            dependent["cur"] = 1
            dependent["key"] = 1
            dependent["tf"] = 1
            new.attrib["Dep"] = ",".join(map(str, sorted(dependent)))    

    if map_type != 0:
        new.attrib["Programming"] = programming       # copy "programming" attribute from source
        new.attrib["LockProgramming"] = lockprogramming
        new.attrib["Mapping"] = str(map_type)
        
        if masterKey != "0":
            enabledValue = "1"

        # add all relevant properties
        for elem in configlock_scale_props:

            prop = et.SubElement(new, "Property",
                                 Name=elem[0],
                                 DName=elem[1],
                                 Map=elem[2],
                                 Api="1" if elem[3] else "0",
                                 Unit="")
            form = et.SubElement(prop, "Format")
            scale = et.SubElement(form, "Scale")

            # if  "PermW" in details.attrib:
            #     tmp.attrib["PermW"] = details.attrib["PermW"]

            if elem[0] == "key.en":
                scale.attrib["Min"] = "0"
                scale.attrib["Max"] = "1"
                scale.attrib["Type"] = "xsd:boolean"
                scale.attrib["Default"] = enabledValue
            if elem[0] == "key.mp" or elem[0] == "key.sp":
                scale.attrib["Min"] = "1"
                scale.attrib["Max"] = "4294967295"
                scale.attrib["Type"] = "xsd:unsignedInt"
                scale.attrib["Off"] = "0"
                if elem[0] == "key.mp":
                    scale.attrib["Default"] = masterKey
                else:
                    scale.attrib["Default"] = serviceKey

            if elem[0] == "key.pu":
                if "Min" in permu.attrib:   # dd file change for some drivers: there is something to copy from dd file
                    scale.attrib["Min"] = permu.attrib["Min"]
                else:
                    scale.attrib["Min"] = "0"
                scale.attrib["Max"] = "4294967295"
                scale.attrib["Type"] = "xsd:unsignedInt"  
                scale.attrib["Default"] = permudef      ## don't use a fixed value, use the default from source to support the import case

            if elem[0] == "key.ps": 
                if "Min" in perms.attrib:    # dd file change for some drivers: there is something to copy from dd file
                    scale.attrib["Min"] = perms.attrib["Min"]
                else:
                    scale.attrib["Min"] = "0"
                scale.attrib["Max"] = "4294967295"
                scale.attrib["Type"] = "xsd:unsignedInt"
                scale.attrib["Default"] = permsdef
            if elem[0] == "key.em":
                scale.attrib["Min"] = "0"
                scale.attrib["Max"] = "1"
                scale.attrib["Type"] = "xsd:boolean"
                scale.attrib["Default"] = isKeyEncrypted(isRaw, masterKey)
            if elem[0] == "key.es":
                scale.attrib["Min"] = "0"
                scale.attrib["Max"] = "1"
                scale.attrib["Type"] = "xsd:boolean"
                scale.attrib["Default"] = isKeyEncrypted(isRaw, serviceKey)
        return new

    sub = getsubdevice(structure, "OTConfig-0")   # Configuration Lock feature based on OTConfig-0 subdevice, e.g. in AA67888.xml

    if sub is not None:
        map_type = 11
        programming = sub.attrib["Programming"]
        lockprogramming = sub.get("LockProgramming", "false")
        configlock_scale_props = [
            ("key.en", "Enable", ""),
            ("key.mp", "Master Password", ""),
            ("key.em", "Encrypted Master Key", "", True)]


        # get and add the dependent features as Dep attribute
        dependent = dict()
        dependent["cur"] = 1
        dependent["key"] = 1
        new.attrib["Dep"] = ",".join(map(str, sorted(dependent)))   

        if isRaw:
            scale = sub.find(".//Property[@PgmName='OTConfig-0:ProgToolPassword']/Format/Scale")
            masterKey = scale.attrib["Default"]
            isEncrypted = isKeyEncrypted(isRaw, masterKey)
 

    else:
        sub = getsubdevice(structure, "Pwd1-1")   # Configuration Lock feature based on Pwd1-1 and Pwd2-1 (AB30663.xml)

        if sub is not None:
            map_type = 12
            programming = sub.attrib["Programming"]
            lockprogramming = sub.get("LockProgramming", "false")
            configlock_scale_props = [
                ("key.en", "Enable", ""),
                ("key.mp", "Master Password", ""),
                ("key.all", "Full or Partial Protection", ""),
                ("key.em", "Encrypted Master Key", "", True)]

            if isRaw:
                subPwd2 = getsubdevice(structure, "Pwd2-1")
                masterKey, isEncrypted, keyAll = getMasterKeyMapping12(sub, subPwd2)

    if map_type != 0:
        new.attrib["Programming"] = programming       # copy "programming" attribute from source
        new.attrib["LockProgramming"] = lockprogramming
        new.attrib["Mapping"] = str(map_type)

        if masterKey != "0":
            enabledValue = "1"

        # add all relevant properties
        for elem in configlock_scale_props:

            prop = et.SubElement(new, "Property",
                                 Name=elem[0],
                                 DName=elem[1],
                                 Map=elem[2],
                                 Api="1",
                                 Unit="")
            form = et.SubElement(prop, "Format")
            scale = et.SubElement(form, "Scale")

            # if  "PermW" in details.attrib:
            #     tmp.attrib["PermW"] = details.attrib["PermW"]

            if elem[0] == "key.en":
                scale.attrib["Min"] = "0"
                scale.attrib["Max"] = "1"
                scale.attrib["Type"] = "xsd:boolean"
                scale.attrib["Default"] = enabledValue

            if elem[0] == "key.mp":
                scale.attrib["Min"] = "1"
                scale.attrib["Max"] = "9999"
                scale.attrib["Type"] = "xsd:unsignedInt"
                scale.attrib["Off"] = "0"
                scale.attrib["Default"] = masterKey

            if elem[0] == "key.all":
                scale.attrib["Min"] = "0"
                scale.attrib["Max"] = "1"
                scale.attrib["Type"] = "xsd:boolean"
                scale.attrib["Default"] = keyAll
            if elem[0] == "key.em":
                scale.attrib["Min"] = "0"
                scale.attrib["Max"] = "1"
                scale.attrib["Type"] = "xsd:boolean"
                scale.attrib["Default"] = isEncrypted

        return new

    if debug > 2:
        print("Info: No Configurationlock feature in the driver.")
    return None

def getPowerSettings(structure, sub, instances, minDimLvl):
    powerSub = getsubdevice(structure, "PowerInfo")
    # now create a Voltage and Power property foreach operating current instance 
    inst = 0
    while sub.find(f".//Property[@Name='cur.{inst}']") is not None:
        if powerSub is not None:
            fProp =et.SubElement(sub, "Property", Name=f"cur.u{inst}", DName="Voltage", Map=f"PowerInfo:Voltage{inst}", Api="1", Unit="V")
            fmt = powerSub.find(f".//Property[@PgmName='PowerInfo:Voltage{inst}']/Format")
            fProp.append(deepcopy(fmt))
            fProp =et.SubElement(sub, "Property", Name=f"cur.pw{inst}", DName="Power", Map="", Api="1", Unit="W")
            fmt = powerSub.find(f".//Property[@PgmName='PowerInfo:Power{inst}']/Format")
            fProp.append(deepcopy(fmt))
            isCreated = ddfile.createFeaPropertyFromDDProperty(sub, powerSub, f"PowerInfo:LoadPower{inst}", f"cur.lpw{inst}")
            if isCreated:
                feaProp = createFeaProperty(f"cur.mdl", "Min Dim Lvl", "", "%", "Scale", instances, Default=minDimLvl, Min="0", Max="100", Type="xsd:float")
                sub.append(feaProp)
        else:
            feaProp = createFeaProperty(f"cur.u{inst}", "Voltage", "", "V", "Scale", instances, Default="-1", Min="-1", Max="-1", Type="xsd:unsignedShort")
            sub.append(feaProp)
            feaProp = createFeaProperty(f"cur.pw{inst}", "Power", "", "W", "Scale", instances, Default="-1", Min="-1", Max="-1", Type="xsd:unsignedShort")
            sub.append(feaProp)
        inst += 1
    return

def detectPwmConfig0PwmCalculationMapping(structure):
    mapping = "5"
    prop = structure.find(".//Property[@PgmName='PwmOut-0:Offset']")
    if prop is not None:
        mapping = "33"
    return mapping

def getsettings_current(structure, switches=None, constants=None, filename=""):
    new = et.Element("Feature", Name="cur", DName="Output Current", Instances="1")
    dep = detectDependencies(structure, "cur", False, key="OTConfig-0|PwmConfig-0", tf="OTConfig-1|PwmConfig-0")
    if dep != "cur":
        new.attrib["Dep"] = dep

    instances = 1
    sub = getsubdevice(structure, "OTConfig-0")   # current feature based on OTConfig-0, e.g. in "aa67888"
    if sub is not None:
        new.attrib["Mapping"] = "1"
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from souce
        copy_LockProgramming(sub, new)                              # same for LockProgramming

        details = getproperty(sub, "OTConfig-0:Command-LEDsetEnable")
        en = et.SubElement(new, "Property", DName="Enable", Name="cur.en", Map="", Api="1", Unit="")
        en.append(copy_format_element(details.find("./Format"), 1))

        invert_bool(en[0][0], "Default")            # cur.en must be the inverse of "ledset enable"
        if en[0][0].attrib["Min"] == en[0][0].attrib["Max"]:
            invert_bool(en[0][0], "Min")                # same for min/max
            invert_bool(en[0][0], "Max")

        details = getproperty(sub, "OTConfig-0:NominalLEDCurrent")
        se = et.SubElement(new, "Property", DName="Current", Name="cur.0", Map="OTConfig-0:NominalLEDCurrent", Api="1", Unit="mA")
        se.append(copy_format_element(details.find("./Format"), 1))

    if sub is None:
        sub = getsubdevice(structure, "OTConfig-1")   # current feature based on the OTConfig-1 subdevice
        if sub is not None:
            new.attrib["Mapping"] = "2"
            new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
            copy_LockProgramming(sub, new)                              # same for LockProgramming

            details = getproperty(sub, "OTConfig-1:ModeSetting-EnableLEDset2Interface")
            en = et.SubElement(new, "Property", DName="Enable", Name="cur.en", Map="", Api="1", Unit="")
            en.append(copy_format_element(details.find("./Format"), 1))

            invert_bool(en[0][0], "Default")            # cur.en must be the inverse of "ledset enable"
            if en[0][0].attrib["Min"] == en[0][0].attrib["Max"]:
                invert_bool(en[0][0], "Min")            # same for min/max
                invert_bool(en[0][0], "Max")
                # handle firmware bug in Outdoor 1DIM APAC and Outdoor 1DIM APAC (DTC) AM22822
                # Those devices do not have an LEDSet interface but fw returns OTConfig-1:ModeSetting-EnableLEDset2Interface = 1
                if en[0][0].attrib["Max"] == "1" and en[0][0].attrib["Default"] == "0":
                    p = getproperty(sub, "OTConfig-1:LEDset2Status-InterfacePort")
                    if p[0][0].attrib["Default"] == "0":
                        en[0][0].attrib["Default"] = "1"

            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

            details = getproperty(sub, "OTConfig-1:DefaultOpCurrent")
            se = et.SubElement(new, "Property",
                               DName="Current",
                               Name="cur.0",
                               Map="OTConfig-1:DefaultOpCurrent",
                               Api="1",
                               Unit="mA")
            se.append(copy_format_element(details.find("./Format"), 1))

            if "PermW" in details.attrib:
                se.attrib["PermW"] = details.attrib["PermW"]

    if sub is None:
        sub = getsubdevice(structure, "OTConfig-2")   # current feature based on the OTConfig-2 subdevice
        if sub is not None:
            dep = detectDependencies(structure, "cur", False, tp="ThermProt-2")
            if dep != "cur":
                new.attrib["Dep"] = dep
            it = constants.find(".//Constant[@Name='NfcType']")
            if it is not None:
                nfcversion = it.attrib["Value"]
            else:
                nfcversion = "0"

            if nfcversion == "3":
                new.attrib["Mapping"] = "24"    # in this case a differnt mapping has to be applied (calc PWM cycle)
                nfc3 = True
            else:
                new.attrib["Mapping"] = "3"     # all other cases
                nfc3 = False

            new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
            copy_LockProgramming(sub, new)                              # same for LockProgramming

            details = sub.find(".//Property[@PgmName='OTConfig-2:ModeSetting-EnableLEDset2Interface']")

            en = et.SubElement(new, "Property",
                               DName="Enable",
                               Name="cur.en",
                               Map="",              # no direct mapping for cur.en
                               Api="1",
                               Unit="")

            if details != None:  
                en.append(copy_format_element(details.find("./Format"), 1))
                if "PermW" in details.attrib:
                    en.attrib["PermW"] = details.attrib["PermW"]            
                invert_bool(en[0][0], "Default")            # cur.en must be the inverse of "ledset enable"
                if en[0][0].attrib["Min"] == en[0][0].attrib["Max"]:
                    invert_bool(en[0][0], "Min")            # same for min/max
                    invert_bool(en[0][0], "Max")

            else:  # some old dd files had no OTConfig-2:ModeSetting-EnableLEDset2Interface, so create structure for cur.en
                form = et.SubElement(en,"Format")
                scal = et.SubElement(form,"Scale",
                                     Default="1",
                                     Min="1",
                                     Max="1",
                                     Type="xsd:boolean")

            details = getproperty(sub, "OTConfig-2:DefaultOpCurrent")

            se = et.SubElement(new, "Property",
                                DName="Current",
                                Name="cur.0",
                                Map="OTConfig-2:DefaultOpCurrent",
                                Api="1",
                                Unit="mA")
            se.append(copy_format_element(details.find("./Format"), 1))

            if "PermW" in details.attrib:
                se.attrib["PermW"] = details.attrib["PermW"]

            # return new

    if sub is None:
        sub = getsubdevice(structure, "OTConfig-3")   # current feature based on OTConfig-3
        if sub is not None:
            instances = getFeatureInstanceCount(sub, new, constants)
            new.attrib["Mapping"] = "4"
            new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
            copy_LockProgramming(sub, new)                              # same for LockProgramming

            details = getproperty(sub, "OTConfig-3:ModeSetting-EnableLEDset2Interface")
            en = et.SubElement(new, "Property", DName="Enable", Name="cur.en", Map="", Api="1", Unit="")
            en.append(copy_format_element(details.find("./Format"), instances))

            for i in range(instances):
                invert_bool(en[0][i], "Default")            # cur.en must be the inverse of "ledset enable"
                if en[0][i].attrib["Min"] == en[0][i].attrib["Max"]:
                    invert_bool(en[0][i], "Min")                # same for min/max
                    invert_bool(en[0][i], "Max")

            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

            details = getproperty(sub, "OTConfig-3:DefaultOpCurrent")
            se = et.SubElement(new, "Property",
                               DName="Current",
                               Name="cur.0",
                               Map="OTConfig-3:DefaultOpCurrent",
                               Api="1",
                               Unit="mA")
            se.append(copy_format_element(details.find("./Format"), instances))

            if "PermW" in details.attrib:
                se.attrib["PermW"] = details.attrib["PermW"]

    if sub is None:
        sub = getsubdevice(structure, "PwmConfig-0")   # current feature based on PwmConfig-0, e.g. in AM10971.xml (OT FIT drivers)
        if sub is not None:
            new.attrib["Mapping"] = detectPwmConfig0PwmCalculationMapping(structure)
            new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
            copy_LockProgramming(sub, new)                              # same for LockProgramming

            ledsetversion = int(constants.find(".//Constant[@Name='LEDSetVersion']").attrib["Value"])

            details = sub.find(".//Property[@PgmName='PwmConfig-0:EnableLEDset']")   # this property is mandatory in PwmConfig
            if details is None:                                      # (import case) but appears under different name in old files
                details = sub.find(".//Property[@PgmName='PwmConfig-0:LedSetEnable']")  # hence also look for this property name
            pro = et.SubElement(new, "Property", DName="Enable", Name="cur.en", Map="", Api="1", Unit="")

            if "PermW" in details.attrib:
                pro.attrib["PermW"] = details.attrib["PermW"]

            if ledsetversion == 0:                                                                    # no ledset means:
                scale = et.SubElement(pro, "Format")                                                # feature cur is always enabled
                et.SubElement(scale, "Scale", Min="1", Max="1", Default="1", Type="xsd:boolean")    # create new Format element
                # and done with "cur.en"
            else:                                                               # here: cur.en must be the inverse of "ledset enable"
                form = copy_format_element(details.find("./Format"), 1)         # get the source settings
                invert_bool(form[0], "Default")
                if form[0].attrib["Min"] == form[0].attrib["Max"]:
                    invert_bool(form[0], "Min")
                    invert_bool(form[0], "Max")
                pro.append(form)

            details = getproperty(sub, "PwmConfig-0:DefaultOpCurrent")
            se = et.SubElement(new, "Property", DName="Current", Name="cur.0", 
                               Map="PwmConfig-0:DefaultOpCurrent", Api="1", Unit="mA")
            se.append(copy_format_element(details.find("./Format"), 1))

            if "PermW" in details.attrib:
                se.attrib["PermW"] = details.attrib["PermW"]

    if sub is None:
        sub = getsubdevice(structure, "OpsCur-0")   # current feature based on OpsCur-0, e.g. in 79371.xml (Nafta drivers)
        if sub is not None:
            new.attrib["Mapping"] = "6"
            new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
            copy_LockProgramming(sub, new)                              # same for LockProgramming

            ledsetversion = int(constants.find(".//Constant[@Name='LEDSetVersion']").attrib["Value"])

            details = getproperty(sub, "OpsCur-0:EnableLEDSetMode")
            en = et.SubElement(new, "Property", DName="Enable", Name="cur.en", Map="", Api="1", Unit="")
            #scale = et.SubElement(en, "Format")

            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

            if ledsetversion == 0:    # ledset is not used, i.e. cur feature is always enabled
                scale = et.SubElement(en, "Format")
                et.SubElement(scale, "Scale", Min="1", Max="1", Default="1", Type="xsd:boolean")  
            else:                     # cur.en must be the inverse of "ledset enable"
                tmp = copy_format_element(details.find("./Format"), 1)
                invert_bool(tmp[0], "Default")            
                if tmp[0].attrib["Min"] == tmp[0].attrib["Max"]:
                    invert_bool(tmp[0], "Min")
                    invert_bool(tmp[0], "Max")
                en.append(tmp)

            details = getproperty(sub, "OpsCur-0:OperatingCurrent")
            se = et.SubElement(new, "Property", DName="Current", Name="cur.0", 
                               Map="OpsCur-0:OperatingCurrent", Api="1", Unit="mA")
            se.append(copy_format_element(details.find("./Format"), 1))

            if "PermW" in details.attrib:
                se.attrib["PermW"] = details.attrib["PermW"]

    if sub is None:
        sub = getsubdevice(structure, "TWA-0")   # current feature based on TWA-0
        if sub is not None:
            new.attrib["Mapping"] = "0"
            new.attrib["Programming"] = sub.attrib["Programming"]
            copy_LockProgramming(sub, new)                              # same for LockProgramming
            new.attrib["Instances"] = "1"   # with TWA-0, the current feature can be there only once!

            en = et.SubElement(new, "Property", DName="Enable", Name="cur.en", Map="", Api="1", Unit="")
            fo = et.SubElement(en, "Format")
            et.SubElement(fo, "Scale", Min="1", Max="1", Default="1")

            cu = et.SubElement(new, "Property", DName="Current", Name="cur.0", Map="TWA-0:Nominalcurrent-Led0", Api="1", Unit="mA")
            cu.append(copy_format_element(sub.find(".//Property[@PgmName='TWA-0:Nominalcurrent-Led0']/Format"), 1))

            cu = et.SubElement(new, "Property", DName="Current", Name="cur.1", Map="TWA-0:Nominalcurrent-Led1", Api="1", Unit="mA")
            cu.append(copy_format_element(sub.find(".//Property[@PgmName='TWA-0:Nominalcurrent-Led1']/Format"), 1))

    if sub is not None:
        hw_resouces = sub.findall("./HwResource")
        for hwr in hw_resouces:
            new.append(deepcopy(hwr))

        # copy the optional constants
        # finding constants is surrounded by try/except, as not all must be present
        try:
            details = structure.find(".//Property[@PgmName='NominalPwmCycle']/Format/Scale")
            val = details.attrib["Default"]
            et.SubElement(constants, "Constant", Name="NominalPwmCycle", Value=val)
        except (NameError, AttributeError):  # not present, so continue without complaining ...
            pass

        try:
            details = getproperty(sub, "RatedNominalCurrent")
            tmp = details.find("./Format/Scale")
            mini = tmp.attrib["Min"]
            maxi = tmp.attrib["Max"]
            et.SubElement(constants, "Constant", Name="RatedMinNominalCurrent", Value=mini)
            et.SubElement(constants, "Constant", Name="RatedMaxNominalCurrent", Value=maxi)
            # tmp.set("test","quatsch")   # adding attributes also works like that...
        except NameError:
            pass

        try:
            details = getproperty(sub, "RealMinimumOutputCurrent")
            tmp = details.find("./Format/Scale")
            val = tmp.attrib["Default"]
            et.SubElement(constants, "Constant", Name="RealMinimumOutputCurrent", Value=val)
        except NameError:
            pass

        minDimLvl = 0
        try:
            details = getproperty(sub, "MinimumRatedDimmingLevel")
            tmp = details.find("./Format/Scale")
            minDimLvl = tmp.attrib["Default"]
            et.SubElement(constants, "Constant", Name="MinimumRatedDimmingLevel", Value=minDimLvl)
        except NameError:
            pass

        try:
            details = getproperty(sub, "MaxJoinedCurrent")
            tmp = details.find("./Format/Scale")
            val = tmp.attrib["Default"]
            et.SubElement(constants, "Constant", Name="MaxJoinedCurrent", Value=val)
        except NameError:
            pass

        try:
            details = getproperty(sub, "PhysicalMinLevel")
            tmp = details.find("./Format/Scale")
            val = tmp.attrib["Default"]
            # mapped = str(dali_to_level(int(val)))  # no need to convert...
            et.SubElement(constants, "Constant", Name="PhysicalMinLevel", Value=val)    # unit is %
        except NameError:
            pass

        # finally get the powersettings (Voltage and Power properties)
        getPowerSettings(structure, new, instances, minDimLvl)

        return new

    if debug > 1:
        print("Info: No current feature in the driver.")
    return None


def getsettings_d2d(structure, switches=None, constants=None, filename=""):

    # note: d2d can be enable with all OM. When D2D is enabled, the calculation of the PhyMin is different:
    # RateMinDimLevel must not be taken into account.

    new = et.Element("Feature", Name="d2d", DName="Dim to Dark", Mapping="0", Instances="1")

    cur = getsubdevice(structure, "D2D-0")   # d2d feature based on D2D-0, e.g. in AM12229
    if cur is not None:
        d2d = cur.find("./Property[@PgmName='D2D-0:Config-Enable']")  # this is mandatory prop in D2D-0, so dont check if this exists

        new.attrib["Programming"] = cur.attrib["Programming"]
        copy_LockProgramming(cur, new)                              # same for LockProgramming
        tmp = et.SubElement(new, "Property", DName="Enable", Name="d2d.en", Map="D2D-0:Config-Enable", Api="1", Unit="")
        tmp.append(copy_format_element(d2d.find("./Format"), 1))
        if "PermW" in cur.attrib:
            new.attrib["PermW"] = cur.attrib["PermW"]
        return new

    cur = getsubdevice(structure, "GFM-1")   # d2d feature based on GFM-1 subdevice
    if cur is not None:
        d2d = cur.find("./Property[@PgmName='GFM-1:D2DConfig-Enable']")
        if d2d is not None:

            min = d2d.find(".//Format/Scale").attrib["Min"]
            max = d2d.find(".//Format/Scale").attrib["Max"]
            if min != max:          # only include d2d if settable

                new.attrib["Programming"] = cur.attrib["Programming"]
                copy_LockProgramming(cur, new)                              # same for LockProgramming
                tmp = et.SubElement(new, "Property", DName="Enable", Name="d2d.en", Map="GFM-1:D2DConfig-Enable", Api="1", Unit="")
                tmp.append(copy_format_element(d2d.find("./Format"), 1))

                if "PermW" in cur.attrib:
                    new.attrib["PermW"] = cur.attrib["PermW"]
                return new

    return None


def getsettings_d2w(structure, switches=None, constants=None, filename=""):

    # note: d2w can be enabled only with DT8 or TD/CF color modes

    new = et.Element("Feature", Name="dw", DName="Dim to Warm", Mapping="26", Instances="1")

    cur = getsubdevice(structure, "D2W-0")   # d2w feature based on the D2W-0 subdevice, e.g. in AM28017
    if cur is not None:
        new.attrib["Programming"] = cur.attrib["Programming"]
        copy_LockProgramming(cur, new)                              # same for LockProgramming

        d2w_lparam = {                 # list of all d2w level parameters to be handled automatically and in the same way
            "D2W-0:DimLevel0": ("dw.l0", "Dim Level 0"),
            "D2W-0:DimLevel1": ("dw.l1", "Dim Level 1"),
            "D2W-0:DimLevel2": ("dw.l2", "Dim Level 2"),
            "D2W-0:DimLevel3": ("dw.l3", "Dim Level 3")}

        d2w_cparam = {                 # list of all d2w colour parameters
            "D2W-0:CCTLevel0": ("dw.c0", "Colour Temperature at Dim Level 0"),
            "D2W-0:CCTLevel1": ("dw.c1", "Colour Temperature at Dim Level 1"),
            "D2W-0:CCTLevel2": ("dw.c2", "Colour Temperature at Dim Level 2"),
            "D2W-0:CCTLevel3": ("dw.c3", "Colour Temperature at Dim Level 3")}

        d2we = cur.find("./Property[@PgmName='D2W-0:Enable']")  # add as switch
        if d2we is not None:
            tmp = d2we.find("./Format/Scale")
            val = tmp.attrib["Default"]
            et.SubElement(switches, "Switch", Name="D2W-0:Enable", Map="D2W-0:Enable", SetValue=val)

        # add loop over all properties
        for prop in cur.findall("./Property"):
            name = prop.attrib['PgmName']
            if name in d2w_lparam:
                pro = et.SubElement(new, "Property", Name=d2w_lparam[name][0], DName=d2w_lparam[name][1], Map=name, Api="1", Unit="%")
                details = prop.find("./Format")
                pro.append(copy_format_element(details, 1))

        # add loop over all color properties
        for prop in cur.findall("./Property"):
            name = prop.attrib['PgmName']
            if name in d2w_cparam:
                pro = et.SubElement(new, "Property", Name=d2w_cparam[name][0], DName=d2w_cparam[name][1], Map="", Api="1", Unit="K")
                src = prop.find("./Format/Scale")
                fo = et.SubElement(pro, "Format")

                # convert to kelvin, but dont touch off
                off = src.attrib["Off"]
                default = src.attrib["Default"]
                if default == off:
                    tmp = off
                else:
                    tmp = mirek_to_kelvin(default)
                et.SubElement(fo, "Scale",
                              Default= tmp,
                              Min= mirek_to_kelvin(src.attrib["Max"]),
                              Max= mirek_to_kelvin(src.attrib["Min"]),
                              Off= "65535")  
        return new

    if debug > 1:
        print("Info: No d2w feature in the driver.")
    return None

# Detects feature dependencies.
# structure - the dd file xml
# self - its own feature name
# depDali - bool to check for dali dependency
# kwargs - multiple subdevice dependencies, <fea feature name>="subdevice1|subdevice2"
# returns the dep string value 
def detectDependencies(structure, self, depDali, **kwargs):
    dependent = dict()
    dependent[self] = 1
    for key, value in kwargs.items():
        subs = value.split('|')
        for sub in subs:
            if ":" in sub:      # its a prop
                if structure.find(f".//Property[@PgmName='{sub}']") is not None:
                    dependent[key] = 1
            else:
                if structure.find(f".//SubDevice[@PgmName='{sub}']") is not None:
                    dependent[key] = 1
    if depDali:
        detectDaliDependencies(structure, None, dependent)
    dep = ",".join(map(str, sorted(dependent)))
    return dep


def detectDaliAddressFeature(sub):
    scales = sub.findall(".//Property[@PgmName='Address']/Format/Scale")
    if scales is not None and len(scales) > 0:
        if scales[0].attrib["Min"] == "0":
            return True
    return False

# Detects all Dali feature dependensies
# structure - the dd file xml
# dalisub - optional daliProperties xml element
# dependent - dependensy dictionary
def detectDaliDependencies(structure, dalisub, dependent):
    if dalisub is None:
        dalisub = structure.find(".//DALIProperties")
    if dalisub is None:
        return

    dependent["da"] = 1
    if detectDaliAddressFeature(dalisub):
        dependent["dadr"] = 1
    if structure.find(".//SubDevice[@PgmName='TDiCorridor-0']") is not None or \
            structure.find(".//SubDevice[@PgmName='TDiCorridor-1']") is not None or \
            structure.find(".//SubDevice[@PgmName='TDiCorridor-5']") is not None:
        dependent["td"] = 1
    if structure.find(".//SubDevice[@PgmName='Emergency-0']") is not None:
        dependent["em"] = 1
    if dalisub.find(".//Property[@PgmName='RGBWAFEnabledChannels']") is not None:
        dependent["rgbw"] = 1
    return


daliDependencies = ""

def getsettings_dali(structure, switches=None, constants=None, filename=""):

    devconfig = structure.find(".//Property[@PgmName='GFM-1:DeviceConfig']")
    if devconfig is not None:
        tmp = devconfig.find("./Format/Scale")
        val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="GFM-1:DeviceConfig", Map="GFM-1:DeviceConfig", SetValue=val)

    dali_param = {                 # list of dali parameters to be handled automatically and in the same way
        "FadeRate": ("da.fr", "Fade Rate", "faderate", "steps/s"),
        "DimmingCurve": ("da.dc", "DimmingCurve", "scale", ""),
        "MinLevel": ("da.mi", "Min Level", "scale", ""),
        "MaxLevel": ("da.ma", "Max Level", "scale", ""),
        "PowerOnLevel": ("da.po", "Power On Level", "scale", ""),
        "SystemFailureLevel": ("da.sf", "System Failure Level", "scale", ""),
        # "PowerOnColour":           ("da.poc","Power On Colour Temperature","scale"),
        # "SysFailureColour":        ("da.sfc","System Failure Colour Temperature","scale"),
        # "CtPhysCoolest":           ("da.pcc","Physical Coldest Colour Temperature","scale"),
        # "CtPhysWarmest":           ("da.pwc","Physical Warmest Colour Temperature","scale"),
        # "CtCoolest":               ("da.cc","Coldest Colour Temperature","scale"),
        # "CtWarmest":               ("da.wc","Warmest Colour Temperature","scale")
    }

    dali_ignore = ["LowGroup", "HighGroup", "Scene0", "Scene1", "Scene2", "Scene3", "Scene4", "Scene5", "Scene6",
                    "Scene7", "Scene8", "Scene9", "Scene10", "Scene11", "Scene12", "Scene13", "Scene14", "Scene15",
                    "Scene16", "Address", "MinLevelDummy", "ExtendedFadeTimeMultiplier",
                    "RGBWAFControl", "RGBWAFEnabledChannels"]

    extfade_remap = {
        "1": "0", "2": "0", "3": "0", "4": "0", "5": "0", "6": "0", "7": "0", "8": "0",
        "9": "0", "10": "0", "11": "0", "12": "0", "13": "0", "14": "0", "15": "0",
        "32": "25", "48": "41", "64": "53", "65": "59"}

    new = et.Element("Feature", Name="da", DName="DALI Settings", Mapping="10", Instances="1")
    dali_props_added = False

    dx = structure.find(".//SubDevice[@PgmName='DexalPSU-0']")
    if dx is not None:
        new.attrib["DName"] = "Dexal Settings" 

    sub = structure.find(".//DALIProperties")
    if sub is not None:
        # check for dependent features of this one:
        dependent = dict()
        detectDaliDependencies(structure, sub, dependent)
        global daliDependencies
        daliDependencies = ",".join(map(str, sorted(dependent)))
        new.attrib["Dep"] = daliDependencies

        maxinstances = getFeatureInstanceCount(sub, new, constants)

        if "Programming" in sub.attrib:        # import case: in some old dd files, the dali programming attrib was missing
            prog = sub.attrib["Programming"]
            lockprogramming = sub.get("LockProgramming","false")
        else:
            prog = "true"                       # then use a "reasonable" default (must be true as Part Prog is not yet supported)
            lockprogramming = "false"
        new.attrib["Programming"] = prog        # copy "programming" attribute from source
        new.attrib["LockProgramming"] = lockprogramming


        # check the fade capability, the selected "fade mode" and get all fade settings of the instance nth of the driver 
        f = []
        for instance in range(maxinstances):
            
            std_fade_time = 0
            fast_fade_time = 0
            fast_fade_min = 0
            fade_capability = 0

            #fade = sub.find("./Property[@PgmName='FadeTime']/Format/Scale")
            fade = get_instance_prop(sub,"FadeTime", instance, maxinstances)
            get_instance_prop
            if fade is not None:
                fade_capability = 1
                std_fade_time = int(fade.attrib["Default"])

            #fade = sub.find("./Property[@PgmName='FastFadeTime']/Format/Scale")
            fade = get_instance_prop(sub,"FastFadeTime", instance, maxinstances)
            if fade is not None:
                fade_capability = 2
                fast_fade_time = int(fade.attrib["Default"])
                fast_fade_min = int(fade.attrib["Min"])
                if fast_fade_min < 1 or fast_fade_min > 27:       # correct illegal "minimum fft"
                    fast_fade_min = 27

            fade_extbase = 0
            fade_extmult = 0
            #fade = sub.find("./Property[@PgmName='ExtendedFadeTimeBase']/Format/Scale")
            fade = get_instance_prop(sub,"ExtendedFadeTimeBase", instance, maxinstances)
            if fade is not None:
                fade_extbase = int(fade.attrib["Default"])
                #fade_extmult = int(sub.find("./Property[@PgmName='ExtendedFadeTimeMultiplier']/Format/ExpValues").attrib["Default"])
                fade_extmult = int(get_instance_prop(sub,"ExtendedFadeTimeMultiplier", instance, maxinstances).attrib["Default"])
                
            if fade_extmult == 0:
                extfade_default = 0         # special case: fadetime is zero for multiplier=0, independent of extbase
            else:
                extfade_default = fade_extmult * 16 + fade_extbase

            fprops={}
            fprops["fade_capability"]= fade_capability
            fprops["std_fade_time"]  = std_fade_time
            fprops["fast_fade_time"] = fast_fade_time
            fprops["fast_fade_min"]  = fast_fade_min
            fprops["extfade_default"]= extfade_default
            f.append(fprops)

        for prop in sub.findall("./Property"):

            dali_props_added = True
            name = prop.attrib["PgmName"]
            pro = None

            if name == "FadeTime" and f[0]["fade_capability"] == 1:     # create a new enum property for the "Standard Fade only" case
                                                                        # this code assumes that multi instance drivers have the same 
                                                                        # fade capablities on each channel

                pro = et.SubElement(new, "Property", Name="da.ft", DName="Fade Time", Map="", Api="1", Unit="s")
                form = et.SubElement(pro, "Format")

                for instance in range(maxinstances):
                    default = f[instance]["std_fade_time"]
                    if default > 0:         # 0 remains 0, but
                        default += 27       # larger values are "mapped"

                    enum = et.SubElement(form, "EnumValues", Default=str(default), Min="0", Max="42", Ref="fadetime")

                    if maxinstances > 1:
                        enum.attrib["Instance"] = str(instance)

            elif name == "FastFadeTime":  # create enum property with "Standard Fades" and "Fast Fade Times"
                fft_values = [0, 0.025, 0.05, 0.075, 0.1, 0.125, 0.15, 0.175, 0.2, 0.225, 0.25,     # fft
                            0.275, 0.3, 0.325, 0.35, 0.375, 0.4, 0.425, 0.45, 0.475, 0.5,           # fft
                            0.525, 0.55, 0.575, 0.6, 0.625, 0.65, 0.675,                            # ttf
                            0.7, 1, 1.4, 2, 2.8, 4, 5.7, 8, 11.3, 16, 22.6, 32, 45.3, 64, 90.5]     # std fade times

                pro = et.SubElement(new, "Property", Name="da.ft", DName="Fade Time", Map="", Api="1", Unit="s")
                form = et.SubElement(pro, "Format")

                for instance in range(maxinstances):
                    if f[instance]["std_fade_time"] > 0:    # standard fade time has priority
                        default = f[instance]["std_fade_time"] + 27
                    else:
                        default = f[instance]["fast_fade_time"]

                    enums = et.SubElement(form, "EnumValues",
                                        Default=str(default),
                                        Min="0",
                                        Max="42")

                    sub = et.SubElement(enums, "EnumValue", Label="0", Value="0")    # add first EnumValue for 0 and then
                    for i in range(f[instance]["fast_fade_min"], 43):                # all others starting with min_fft
                        et.SubElement(enums, "EnumValue", Label=str(fft_values[i]), Value=str(i))

                    if maxinstances > 1:
                        enums.attrib["Instance"] = str(instance)

            elif name == "ExtendedFadeTimeBase":  # create new enum property for Ext Fade Feature 

                pro = et.SubElement(new, "Property", Name="da.ef", DName="Extended Fadetime", Map="", Api="1", Unit="s")
                form = et.SubElement(pro, "Format")

                for instance in range(maxinstances):

                    setting = str(f[instance]["extfade_default"])      # in case one of the duplicated entries is used, remap it to the supported ones
                    if setting in extfade_remap:
                        setting = extfade_remap[setting]
                    enum = et.SubElement(form, "EnumValues", Default=setting, Min="0", Max="79", Ref="extfade")

                    if maxinstances > 1:
                        enum.attrib["Instance"] = str(instance)

            elif name == "OperatingMode":    # found a switch
                tmp = prop.find("./Format/EnumValues")
                val = tmp.attrib["Default"]
                et.SubElement(switches, "Switch", Name="OperatingMode", Map="OperatingMode", SetValue=val)


            elif name == "PhysicalMinLevel":  # found a constant
                tmp = prop.find("./Format/Scale")
                value = tmp.attrib["Default"]
                co = et.Element("Constant", Name="PhysicalMinLevel", Value=value)   # unit is %
                constants.append(co)


            elif name in dali_param:   # all other DALI properties
                pro = et.SubElement(new, "Property", Name=dali_param[name][0], DName=dali_param[name][1], 
                                    Map=name, Api="1", Unit=dali_param[name][3])

                if dali_param[name][2] == "scale":
                    form = prop.find("./Format")

                    for scale in form:     # correction of dd file inconsistency: 255 is an "illegal" DALI value; add Off instead.
                        max = scale.attrib["Max"]
                        if max == "255":
                            scale.attrib["Max"] = "254"
                            scale.attrib["Off"] = "255"

                    pro.append(copy_format_element(form, maxinstances))
                else:
                    src = prop.find("./Format/Scale")
                    form = et.SubElement(pro, "Format")

                    for instance in range(maxinstances):
                        value = get_instance_value(prop, instance, maxinstances)

                        enum = et.SubElement(form, "EnumValues",
                                            Ref=dali_param[name][2],
                                            Default=value,
                                            Min=src.attrib["Min"],
                                            Max=src.attrib["Max"])
                        if maxinstances > 1:
                            enum.attrib["Instance"] = str(instance)

            else:    # anything left? hopefully not, else cry!
                if debug > 0:
                    if name not in dali_ignore:
                        print("Warning: found an unhandled DaliProperty: ", name)

            if "PermW" in prop.attrib and pro is not None:     # just in case there is a PermW value for the DALI properties, add it
                pro.set("PermW", prop.attrib["PermW"])   # (does not apply for const & switches)

        new.attrib["Instances"] = str(maxinstances)

        # dont return here, there may be more subdevices with daliproperties 

    sub = getsubdevice(structure, "DimmingA-0")     # this subdevice has some DALI parameter as well!
    if sub is not None:
        new.attrib["Programming"] = "true"
        new.attrib["LockProgramming"] = "false" 
        new.attrib["Dep"] = "da"                    # no other dependencies in case of this subdevice
        dali_props_added = True

        subsub = sub.find(".//Property[@PgmName='DimmingA-0:MinDimLevel']/Format")
        prop = et.SubElement(new, "Property", Name="da.mi", DName="Min Level", Map="DimmingA-0:MinDimLevel", Api="1", Unit="")
        prop.append(copy_format_element(subsub, 1))

        subsub = sub.find(".//Property[@PgmName='DimmingA-0:DimmingCurve']/Format")
        prop = et.SubElement(new, "Property", Name="da.dc", DName="Dimming Curve", Map="DimmingA-0:DimmingCurve", Api="1", Unit="")
        prop.append(copy_format_element(subsub, 1))

    if dali_props_added==True:    
        return new
    else:
        return None


def getsettings_dali_addressing(structure, switches=None, constants=None, filename=""):
    # dali_ignore = ["LowGroup", "HighGroup", "Scene0", "Scene1", "Scene2", "Scene3", "Scene4", "Scene5", "Scene6",
    #                 "Scene7", "Scene8", "Scene9", "Scene10", "Scene11", "Scene12", "Scene13", "Scene14", "Scene15",
    #                 "Scene16", "Address", "MinLevelDummy",
    #                 "ExtendedFadeTimeMultiplier"]

    new = et.Element("Feature", Name="dadr", DName="DALI Addressing", Mapping="30", Instances="1")
    dali_props_added = False

    sub = structure.find(".//DALIProperties")
    if sub is not None:
        # dependencies are the same as dali
        new.attrib["Dep"] = daliDependencies

        maxinstances = getFeatureInstanceCount(sub, new, constants)

        if "Programming" in sub.attrib:        # import case: in some old dd files, the dali programming attrib was missing
            prog = sub.attrib["Programming"]
            lockprogramming = sub.get("LockProgramming","false")
        else:
            prog = "true"                       # then use a "reasonable" default (must be true as Part Prog is not yet supported)
            lockprogramming = "false"
        new.attrib["Programming"] = prog        # copy "programming" attribute from source
        new.attrib["LockProgramming"] = lockprogramming

        enabledInstanceValues = ['0'] * maxinstances
        # Address property
        scales = sub.findall(".//Property[@PgmName='Address']/Format/Scale")
        if scales is not None and len(scales) > 0:
            if scales[0].attrib["Min"] == "255":
                return None
            dali_props_added = True
            pro = et.SubElement(new, "Property", Name="dadr.adr", DName="DALI Address", Map="Address", Api="1", Unit="")
            form = et.SubElement(pro, "Format")

            for scale in scales:
                value = scale.attrib["Default"]
                enum = et.SubElement(form, "EnumValues",
                                    Default=value,
                                    Ref="daliaddr"
                                    )
                instance = 0
                if maxinstances > 1:
                    instance = scale.attrib["Instance"]
                    enum.attrib["Instance"] = instance
                if value != '255':
                    enabledInstanceValues[int(instance)] = '1'
        # group property
        group = createFeaProperty("dadr.grp", "Group", "", "", "Scale", maxinstances, Default="0", Min="0", Max="65535")
        new.append(group)
        lowgroup = sub.find("./Property[@PgmName='LowGroup']")
        higroup = sub.find("./Property[@PgmName='HighGroup']")
        if lowgroup is not None:
            for instance in range(maxinstances):
                lowvalue = get_instance_value(lowgroup, instance, maxinstances)
                hivalue = get_instance_value(higroup, instance, maxinstances)
                value = (int(hivalue) << 8) + int(lowvalue)
                set_instance_value(group, str(value), instance, maxinstances)
                if value != 0:
                    enabledInstanceValues[instance] = '1'
        # scene property
        szene = createFeaProperty("dadr.sz", "Scenes", "", "", "HexBytes", maxinstances, Length="16", Default="FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF")
        new.append(szene)
        hexinstancevalues = [''] * maxinstances
        for index in range(16):
            scene = sub.find(f"./Property[@PgmName='Scene{index}']")
            if scene is not None:
                for instance in range(maxinstances):
                    scenevalue = get_instance_value(scene, instance, maxinstances)
                    hex = f'{int(scenevalue):0>2X}'
                    hexinstancevalues[instance] += hex
        for instance in range(maxinstances):
            value = hexinstancevalues[instance]
            if value == '':
                value = 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
            set_instance_value(szene, value, instance, maxinstances)
            if value != 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF':
                enabledInstanceValues[instance] = '1'

        # enabled property
        pro = et.SubElement(new, "Property", Name="dadr.en", DName="Enable", Map="", Api="1", Unit="")
        form = et.SubElement(pro, "Format")

        for instance in range(maxinstances):
            enum = et.SubElement(form, "Scale", Default=enabledInstanceValues[instance], Min="0", Max="1")

            if maxinstances > 1:
                enum.attrib["Instance"] = str(instance)

    if dali_props_added==True:    
        return new
    else:
        return None
    
def createColorProperties(structure, subdevice, colorProps, feaFeature):
    sub = structure.find(subdevice)
    if sub is not None:
        if "Programming" in sub.attrib:         # import case: in some old dd files, the dali programming attrib was missing
            prog = sub.attrib["Programming"]
            lockprogramming = sub.get("LockProgramming", "false")
        else:
            prog = "true"                       # then use a "reasonable" default (must be true as Part Prog is not yet supported)
            lockprogramming = "false"
        feaFeature.attrib["Programming"] = prog        # set "programming" attribute as determined from source
        feaFeature.attrib["LockProgramming"] = lockprogramming 
        for prop in sub.findall("./Property"):

            name = prop.attrib["PgmName"]

            if name in colorProps:            # check for color properties, ignore all other here

                src = prop.find("./Format/Scale")

                default = src.attrib["Default"]     # convert value into K unless it is Off 
                if "Off" in src.attrib:
                    if default != src.attrib["Off"]: 
                        default = mirek_to_kelvin(src.attrib["Default"])
                else:
                    default = mirek_to_kelvin(src.attrib["Default"])
                dname = colorProps[name][1]
                if dname == "":
                    dname = ddfile.getDisplayName(prop)
                pro = et.SubElement(feaFeature, "Property", Name=colorProps[name][0], DName=dname, Map=f"{name}|T", Api="1", Unit="K")
                fo = et.SubElement(pro, "Format")
                sc = et.SubElement(fo, "Scale",
                                   Default=default,
                                   Min=mirek_to_kelvin(src.attrib["Max"]),
                                   Max=mirek_to_kelvin(src.attrib["Min"]))
                if "Off" in src.attrib:                 # add Off attribute if it it exists
                    sc.attrib["Off"] = src.attrib["Off"]

            if "PermW" in prop.attrib and prop is not None:     # just in case there is a PermW value for the DALI properties, add it
                prop.set("PermW", prop.attrib["PermW"])   # (does not apply for const & switches)

    return

def getsettings_dali_dt8(structure, switches=None, constants=None, filename=""):

    dali8_param = {                 # list of dali dt8 parameters to be handled automatically and in the same way
        "PowerOnColour": ("d8.poc", "Power On Colour Temperature", "scale"),
        "SysFailureColour": ("d8.sfc", "System Failure Colour Temperature", "scale"),
        "CtPhysCoolest": ("d8.pcc", "Physical Coldest Colour Temperature", "scale"),
        "CtPhysWarmest": ("d8.pwc", "Physical Warmest Colour Temperature", "scale"),
        "CtCoolest": ("d8.cc", "Coldest Colour Temperature", "scale"),
        "CtWarmest": ("d8.wc", "Warmest Colour Temperature", "scale")}

    new = et.Element("Feature", Name="d8", DName="DALI TW Settings", Mapping="27", Instances="1")

    sub = structure.find(".//DALIProperties")
    if sub is not None:

        instances = getFeatureInstanceCount(sub, new, constants)

        if "Programming" in sub.attrib:         # import case: in some old dd files, the dali programming attrib was missing
            prog = sub.attrib["Programming"]
            lockprogramming = sub.get("LockProgramming", "false")
        else:
            prog = "true"                       # then use a "reasonable" default (must be true as Part Prog is not yet supported)
            lockprogramming = "false"
        new.attrib["Programming"] = prog        # set "programming" attribute as determined from source
        new.attrib["LockProgramming"] = lockprogramming 

        dt8_properties_exist = False
        for prop in sub.findall("./Property"):

            name = prop.attrib["PgmName"]

            if name in dali8_param:            # check for DALI DT8 properties, ignore all other here
                dt8_properties_exist = True    # found at least one

                src = prop.find("./Format/Scale")

                default = src.attrib["Default"]     # convert value into K unless it is Off 
                if "Off" in src.attrib:
                    if default != src.attrib["Off"]: 
                        default = mirek_to_kelvin(src.attrib["Default"])
                else:
                    default = mirek_to_kelvin(src.attrib["Default"])

                pro = et.SubElement(new, "Property", Name=dali8_param[name][0], DName=dali8_param[name][1], Map="", Api="1", Unit="K")
                fo = et.SubElement(pro, "Format")
                sc = et.SubElement(fo, "Scale",
                                   Default=default,
                                   Min=mirek_to_kelvin(src.attrib["Max"]),
                                   Max=mirek_to_kelvin(src.attrib["Min"]))
                if "Off" in src.attrib:                 # add Off attribute if it it exists
                    sc.attrib["Off"] = src.attrib["Off"]

            if "PermW" in prop.attrib and prop is not None:     # just in case there is a PermW value for the DALI properties, add it
                prop.set("PermW", prop.attrib["PermW"])   # (does not apply for const & switches)

        if dt8_properties_exist == True:
            return new
        else:
            return None   # no DT8 property in this driver -> don't include this feature at all

    return None

def getsettings_rgbw(structure, switches=None, constants=None, filename=""):
    new = et.Element("Feature", Name="rgbw", DName="RGBWAF", Mapping="0", Instances="1")
    dali_props_added = False

    sub = structure.find(".//DALIProperties")
    if sub is not None:
        # dependencies are the same as dali
        new.attrib["Dep"] = daliDependencies

        prog = "true"
        lockprogramming = "false"
        if "Programming" in sub.attrib:
            prog = sub.attrib["Programming"]
            lockprogramming = sub.get("LockProgramming", "false")

        new.attrib["Programming"] = prog        # copy "programming" attribute from source
        new.attrib["LockProgramming"] = lockprogramming

        # channel property
        rgbChannel = sub.find(".//Property[@PgmName='RGBWAFEnabledChannels']")
        if rgbChannel is not None:
            dali_props_added = True
            dname = ddfile.getDisplayName(rgbChannel)
            pro = et.SubElement(new, "Property", Name="rgbw.chn", DName=dname, Map="RGBWAFEnabledChannels", Api="1", Unit="")
            pro.append(copy_format_element(rgbChannel.find("./Format"), 1))
        # control properties
        rgbCtrl = sub.find(".//Property[@PgmName='RGBWAFControlPowerOn']")
        if rgbCtrl is not None:
            dali_props_added = True
            dname = ddfile.getDisplayName(rgbCtrl)
            pro = et.SubElement(new, "Property", Name="rgbw.pctr", DName=dname, Map="RGBWAFControlPowerOn", Api="1", Unit="")
            pro.append(copy_format_element(rgbCtrl.find("./Format"), 1))
        rgbCtrl = sub.find(".//Property[@PgmName='RGBWAFControlSysFailure']")
        if rgbCtrl is not None:
            dali_props_added = True
            dname = ddfile.getDisplayName(rgbCtrl)
            pro = et.SubElement(new, "Property", Name="rgbw.sctr", DName=dname, Map="RGBWAFControlSysFailure", Api="1", Unit="")
            pro.append(copy_format_element(rgbCtrl.find("./Format"), 1))
        # color properties
        rgbCtrl = sub.find(".//Property[@PgmName='RGBWAFPowerOn']")
        if rgbCtrl is not None:
            dali_props_added = True
            dname = ddfile.getDisplayName(rgbCtrl)
            pro = et.SubElement(new, "Property", Name="rgbw.pcol", DName=dname, Map="RGBWAFPowerOn", Api="1", Unit="")
            pro.append(copy_format_element(rgbCtrl.find("./Format"), 1))
        rgbCtrl = sub.find(".//Property[@PgmName='RGBWAFSysFailure']")
        if rgbCtrl is not None:
            dali_props_added = True
            dname = ddfile.getDisplayName(rgbCtrl)
            pro = et.SubElement(new, "Property", Name="rgbw.scol", DName=dname, Map="RGBWAFSysFailure", Api="1", Unit="")
            pro.append(copy_format_element(rgbCtrl.find("./Format"), 1))
    if dali_props_added==True:    
        return new
    else:
        return None

def getsettings_dexal(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="dx", DName="Dexal PSU", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "DexalPSU-0")   # Dexal PSU feature based on DexalPSU-0 subdevice-0, e.g. in AM18522
    if sub is not None:
        dx = sub.find("./Property[@PgmName='DexalPSU-0:Enable']")  # this is a mandatory prop in dx, so dont check if this exists
        default = dx.find("./Format/Scale").attrib["Default"]

        new.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, new)

        prop = et.SubElement(new, "Property", DName="Enable", Name="dx.en", Map="DexalPSU-0:Enable", Api="1", Unit="")
        form = et.SubElement(prop, "Format")
        et.SubElement(form, "Scale", Default=default, Min="0", Max="1", Type="xsd:boolean")

        if "PermW" in dx.attrib:
            prop.attrib["PermW"] = dx.attrib["PermW"]

        return new

    sub = getsubdevice(structure, "DexalA-0")   # Dexal PSU feature in NAFTA driver, e.g. in 78033
    if sub is not None:
        dx = sub.find("./Property[@PgmName='DexalA-0:Enable']")  # this is a mandatory prop in dx, so dont check if this exists
        default = dx.find("./Format/Scale").attrib["Default"]

        new.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, new)
        prop = et.SubElement(new, "Property", DName="Enable", Name="dx.en", Map="DexalA-0:Enable", Api="1", Unit="")
        form = et.SubElement(prop, "Format")
        et.SubElement(form, "Scale", Default=default, Min="0", Max="1", Type="xsd:boolean")

        if "PermW" in dx.attrib:
            prop.attrib["PermW"] = dx.attrib["PermW"]

        return new

    if debug > 1:
        print("Info: No DX PSU feature in the driver.")
    return None


def getsettings_dimmingmode(structure, switches=None, constants=None, filename=""):
    new = et.Element("Feature", DName="Dimming Mode", Name="dm", Mapping="0", Instances="1")

    cur = getsubdevice(structure, "OTConfig-3")   # dm feature based on OTConfig-3, e.g. in AM18316
    if cur is not None:
        dm = cur.find("./Property[@PgmName='OTConfig-3:PWM-Threshold']")
        if dm is not None:        
            details = dm.find("./Format/Scale")
            min = details.attrib["Min"]
            max = details.attrib["Max"]
            if min!=max:        # property exists and value is changable

                new.attrib["Programming"] = cur.attrib["Programming"]       # copy "programming" attribute from souce
                copy_LockProgramming(cur, new)                              # LockProgramming as well
                # see open issues !!! tbd: PermW  %%%

                prop = et.SubElement(new, "Property", DName="Enable", Name="dm.en", Map="OTConfig-3:PWM-Threshold", Api="1", Unit="")
                prop.append(copy_format_element(dm.find("./Format"), 1))

                if "PermW" in dm.attrib:
                    prop.attrib["PermW"] = dm.attrib["PermW"]

                # when the dimming mode feature exists, then add constant for Dimming Mode (mandatory with DM feature)
                tmp = cur.find("./Property[@PgmName='PWM-AnalogDimmingThresholdCurrent']/Format/Scale")
                if tmp is None:      # import case: in some older driver files, this constant is stored under a different name
                    tmp = cur.find("./Property[@PgmName='OTConfig-3:PWM-AnalogDimmingThresholdCurrent']/Format/Scale")
                value = tmp.attrib["Default"]
                elem = et.Element("Constant", Name="PWM-AnalogDimmingThresholdCurrent", Value=value)
                constants.append(elem)
                return new

    cur = getsubdevice(structure, "TWA-0")   # dimming mode feature based on TWA-0 subdevice, e.g. in AM28015
    if cur is not None:
        dm = cur.find("./Property[@PgmName='TWA-0:PWM-Threshold']")
        if dm is not None:

            new.attrib["Programming"] = cur.attrib["Programming"]       # copy "programming" attribute from source
            copy_LockProgramming(cur, new)

            prop = et.SubElement(new, "Property", DName="Enable", Name="dm.en", Map="TWA-0:PWM-Threshold", Api="1", Unit="")
            prop.append(copy_format_element(dm.find("./Format"), 1))

            if "PermW" in dm.attrib:
                prop.attrib["PermW"] = dm.attrib["PermW"]

            # add constant for DM feature
            tmp = cur.find("./Property[@PgmName='PWM-AnalogDimmingThresholdCurrent']/Format/Scale")
            if tmp is None:      # import case: in some driver, this constant is hidden somewhere else
                tmp = cur.find("./Property[@PgmName='TWA-0:PWM-AnalogDimmingThresholdCurrent']/Format/Scale")
            value = tmp.attrib["Default"]
            elem = et.Element("Constant", Name="PWM-AnalogDimmingThresholdCurrent", Value=value)
            constants.append(elem)
            return new

    if debug > 1:
        print("Info: No dimming mode feature in the driver.")
    return None

def custom_round(x, base=5):
    return base * round(x/base)

def getsettings_driverguard(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="dg", DName="Driver Guard", Mapping="17", Instances="1")

    sub = getsubdevice(structure, "DG-0")   # Driver Guard feature based on DG-0, e.g. in AM29183
    if sub is not None:
        tmp = sub.find("./Property[@PgmName='DG-0:Enable-PermanentThermalShutdown']")
        if tmp is not None:
            new.attrib["Dep"] = "dg, dgps"
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)

        # note: this enable byte must be 0 or 3, but the property "dg.en"
        # shall be consistent with other enable properties, i.e. use only 0 or 1
        tmp = sub.find("./Property[@PgmName='DG-0:Enable']")
        default = int(tmp.find("./Format/Scale").attrib["Default"])
        if default == 3:
            default = 1

        if "PermW" in tmp.attrib:
            perm = tmp.attrib["PermW"]
        else:
            perm = None

        t1 = et.SubElement(new, "Property", DName="Enable", Name="dg.en", Map="", Api="1", Unit="")
        t2 = et.SubElement(t1, "Format")
        t3 = et.SubElement(t2, "Scale", Default=str(default), Min="0", Max="1", Type="xsd:boolean")
        if perm is not None:
            t1.attrib["PermW"] = perm

        tmp = sub.find("./Property[@PgmName='DG-0:PrestartDerating']")
        t1 = et.SubElement(new, "Property", DName="Prestart Derating", Name="dg.pre", Map="DG-0:PrestartDerating", Api="1", Unit="°C")
        if perm is not None:
            t1.attrib["PermW"] = perm
        t2 = et.SubElement(t1, "Format")
        t3 = et.SubElement(t2, "EnumValues",
                           Default=str(custom_round(int(tmp.find("Format/Scale").attrib["Default"]))) ,
                           Min=tmp.find("Format/Scale").attrib["Min"],
                           Max=tmp.find("Format/Scale").attrib["Max"],
                           Ref="prestartderating"
                           )

        tmp = sub.find("./Property[@PgmName='DG-0:DeratingLevel']")
        t1 = et.SubElement(new, "Property", DName="DeratingLevel", Name="dg.dl", Map="DG-0:DeratingLevel", Api="1", Unit="%")
        if perm is not None:
            t1.attrib["PermW"] = perm
        t1.append(copy_format_element(tmp.find("./Format"), 1))

        # optional property with special handling
        powerderating = sub.find("./Property[@PgmName='DG-0:PowerDeratingLevel']/Format/Scale")
        powermin_rel = int(powerderating.attrib["Min"])
        powermax_rel = int(powerderating.attrib["Max"])
        powerdef_rel = int(powerderating.attrib["Default"])

        value = sub.find("./Property[@PgmName='RatedPower']/Format/Scale").attrib["Default"]
        power_absolute = int(value)

        elem = et.Element("Constant", Name="MaxPower", Value=value)
        constants.append(elem)

        if not (powermin_rel == powermax_rel):    # property is settable, i.e. include it into list of properties

            t1 = et.SubElement(new, "Property", DName="Power Derating", Name="dg.pd", Map="", Api="1", Unit="W")
            t2 = et.SubElement(t1, "Format")
            t3 = et.SubElement(t2, "Scale", Min=str(int(round(powermin_rel * power_absolute * 0.0001, 0))),
                               Max=str(int(round(powermax_rel * power_absolute * 0.0001, 0))),
                               Default=str(int(round(powerdef_rel * power_absolute * 0.0001, 0))),
                               Type="xsd:unsignedInt")
            if perm is not None:
                t1.attrib["PermW"] = perm

        # copy the constants ? --> there are none in Driver Guard

        return new

    if debug > 1:
        print("Info: no DriverGuard feature in the driver.")
    return None

def getsettings_permShutdown(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="dgps", DName="Permanent Shutdown", Mapping="31", Instances="1", Dep="dg,dgps")

    sub = getsubdevice(structure, "DG-0")   # Driver Guard feature based on DG-0, e.g. in AM29183
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)

        tmp = sub.find("./Property[@PgmName='DG-0:Enable-PermanentThermalShutdown']")
        if tmp is not None:
            l1 = int(tmp.find("./Format/EnumValues").attrib["Default"])
            enable = (l1 << 2) # get the byte value according the dd file bit offsets
            default = 0
            if enable == 0xA4:
                default = 1

            if "PermW" in tmp.attrib:
                perm = tmp.attrib["PermW"]
            else:
                perm = None

            t1 = et.SubElement(new, "Property", DName="Enable", Name="dgps.en", Map="", Api="1", Unit="")
            t2 = et.SubElement(t1, "Format")
            t3 = et.SubElement(t2, "Scale", Default=str(default), Min="0", Max="1", Type="xsd:boolean")
            if perm is not None:
                t1.attrib["PermW"] = perm

            return new

    if debug > 1:
        print("Info: no DriverGuard Permanent Shutdown feature in the driver.")
    return None


def getsettings_emergency(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="em", DName="Emergency", Mapping="16", Instances="1")

    sub = getsubdevice(structure, "Emergency-0")   # Emergency feature based on Emergency-0, e.g. in AM28411
    if sub is not None:

        # check for dependent features of this one:
        dependent = detectDependencies(structure, "em", True)
        new.attrib["Dep"] = dependent
        new.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, new)

        # determine number of instances for this feature "em"
        instances = getFeatureInstanceCount(sub, new, constants)

        emplevel = sub.find("./Property[@PgmName='Emergency-0:DCLightLevel']")  # this is mandatory prop in Emergency-0, so dont check if this exists

        isNonDaliFeature = get_bool_value(sub, "Emergency-0:NoDali", False)
        emlevel = sub.find("./Property[@PgmName='Emergency-0:DCLightLevel']/Format")  

        ena = et.SubElement(new, "Property", DName="Enable", Name="em.en", Map="", Api="1", Unit="")
        form1 = et.SubElement(ena, "Format")

        dname = "DALI Level"
        unit = ""
        if isNonDaliFeature:
            dname = get_display_name(sub, "Emergency-0:DCLightLevel")
            unit = get_unit(sub, "Emergency-0:DCLightLevel")

        lev = et.SubElement(new, "Property", DName=dname, Name="em.set", Map="", Api="1", Unit=unit)
        form2 = et.SubElement(lev, "Format")

        sources = emlevel.findall("./Scale")
        index = 0
        for source in sources:      # add as many scale elements as instances, but only if instances > 1

            defaultlevel = source.attrib["Default"]
            maxlevel = source.attrib["Max"]
            minlevel = source.attrib["Min"]

        #for index in range(instances):    
            if defaultlevel == "255":
                tmp = et.SubElement(form1, "Scale", Default="0", Min="0", Max="1", Type="xsd:boolean")   # em is disabled
            else:
                tmp = et.SubElement(form1, "Scale", Default="1", Min="0", Max="1", Type="xsd:boolean")   # any other value means "enabled"
            if instances > 1:
                tmp.attrib["Instance"] = str(index)

            if maxlevel == "255":  # 255 is off, thus max for level is 254
                maxlevel = "254"   # if max is smaller, dont change it
            if defaultlevel == "255":
                defaultlevel = maxlevel

            tmp = et.SubElement(form2, "Scale",
                                Default=defaultlevel,
                                Min=minlevel,
                                Max=maxlevel,
                                Off="255",
                                Type="xsd:unsignedByte")
            if instances > 1:
                tmp.attrib["Instance"] = str(index)
            index += 1

        emlock = sub.find("./Property[@PgmName='Emergency-0:ConfigLock-DaliParamLock']")  # this is mandatory prop in Emergency-0, so don't check if this exists
        dname = "Lock DALI Parameter"
        if isNonDaliFeature:
            dname = get_display_name(sub, "Emergency-0:ConfigLock-DaliParamLock")
        lock = et.SubElement(new, "Property", DName=dname, Name="em.ld",
                             Map="Emergency-0:ConfigLock-DaliParamLock", Api="1", Unit="")
        lock.append(copy_format_element(emlock.find("./Format"), instances))

        noDaliProp = createFeaProperty("em.nd", "No DALI", "", "", "Scale", 1, Default="1" if isNonDaliFeature else "0", Min="0", Max="1", Type="xsd:boolean")
        new.append(noDaliProp)

        if "PermW" in emplevel.attrib:
            ena.attrib["PermW"] = emplevel.attrib["PermW"]
            lev.attrib["PermW"] = emplevel.attrib["PermW"]
            lock.attrib["PermW"] = emplevel.attrib["PermW"]

        return new

    if debug > 1:
        print("Info: no EM feature in the driver.")
    return None


def getsettings_eol(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="eol", DName="End of Life", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "EOL-0")   # End of Life feature based on EOL-0, e.g. in AM28411
    if sub is not None:
        dep = detectDependencies(structure, "eol", False, pod="EOL-0:Command-HoldOnEnable", spd="EOL-0:Command-SPDEnable")
        if dep != "eol":
            new.attrib["Dep"] = dep
        sea = sub.find("./Property[@PgmName='EOL-0:Command-EOLEnable']")  # this  is mandatory in EOL-0, so dont check if it exists
        sti = sub.find("./Property[@PgmName='EOL-0:EOLTime']")  # this  is mandatory in EOL-0, so dont check if it exists

        new.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, new)

        ten = et.SubElement(new, "Property", DName="Enable", Name="eol.en", Map="EOL-0:Command-EOLEnable", Api="1", Unit="")
        ten.append(copy_format_element(sea.find("./Format"), 1))
        tti = et.SubElement(new, "Property", DName="Life Time", Name="eol.ti", Map="EOL-0:EOLTime", Api="1", Unit="kh")
        tti.append(copy_format_element(sti.find("./Format"), 1))

        if "PermW" in sea.attrib:
            ten.attrib["PermW"] = sea.attrib["PermW"]
            tti.attrib["PermW"] = sea.attrib["PermW"]
        return new

    sub = getsubdevice(structure, "EOLA-0")   # End of Life feature based on EOLA-0, e.g. in 79371
    if sub is not None:
        sea = sub.find("./Property[@PgmName='EOLA-0:Enable']")  # this  is mandatory in EOLA-0, so no need to check if it exists
        sti = sub.find("./Property[@PgmName='EOLA-0:EOLTime']")  # this  is mandatory in EOLA-0, so no need to check if it exists

        new.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, new)

        ten = et.SubElement(new, "Property", DName="Enable", Name="eol.en", Map="EOLA-0:Enable", Api="1", Unit="")
        ten.append(copy_format_element(sea.find("./Format"), 1))
        tti = et.SubElement(new, "Property", DName="Time", Name="eol.ti", Map="EOLA-0:EOLTime", Api="1", Unit="kh")
        tti.append(copy_format_element(sti.find("./Format"), 1))

        # no permissions used in NAFTA drivers, so dont try to copy them here

        return new

    if debug > 1:
        print("Info: No eol feature in driver.")
    return None

def getsettings_pod(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="pod", DName="Power-On Delay", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "EOL-0")   # End of Life feature based on EOL-0, e.g. in AM28411
    if sub is not None:
        he = sub.find("./Property[@PgmName='EOL-0:Command-HoldOnEnable']")
        if he is not None:
            dep = detectDependencies(structure, "pod", False, eol="EOL-0", spd="EOL-0:Command-SPDEnable")
            new.attrib["Dep"] = dep
            new.attrib["Programming"] = sub.attrib["Programming"]
            copy_LockProgramming(sub, new)
            ddfile.createFeaPropertyFromDDProperty(new, sub, "EOL-0:Command-HoldOnEnable", "pod.en")
            ddfile.createFeaPropertyFromDDProperty(new, sub, "EOL-0:HoldOnlevel", "pod.lvl")
            ddfile.createFeaPropertyFromDDProperty(new, sub, "EOL-0:HoldOnTime", "pod.t")
            ddfile.createFeaPropertyFromDDProperty(new, sub, "EOL-0:HoldOnFade", "pod.f")
            return new

    if debug > 1:
        print("Info: No pod feature in driver.")
    return None

def getsettings_spd(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="spd", DName="Smart SPD Monitoring", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "EOL-0")   # End of Life feature based on EOL-0, e.g. in AM28411
    if sub is not None:
        he = sub.find("./Property[@PgmName='EOL-0:Command-SPDEnable']")
        if he is not None:
            dep = detectDependencies(structure, "spd", False, eol="EOL-0", pod="EOL-0:Command-HoldOnEnable")
            new.attrib["Dep"] = dep
            new.attrib["Programming"] = sub.attrib["Programming"]
            copy_LockProgramming(sub, new)
            ddfile.createFeaPropertyFromDDProperty(new, sub, "EOL-0:Command-SPDEnable", "spd.en")
            ddfile.createFeaPropertyFromDDProperty(new, sub, "EOL-0:Command-SPDwarning", "spd.wng")
            return new

    if debug > 1:
        print("Info: No spd feature in driver.")
    return None

def getsettings_rpt(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="rpt", DName="Repeater Settings", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "Repeater-0")
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, new)
        ddfile.createFeaPropertyFromDDProperty(new, sub, "Repeater-0:Mode-DT8", "rpt.mo8")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "Repeater-0:Mode-Dali", "rpt.mod")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "Repeater-0:Mode-Input", "rpt.moi")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "Repeater-0:PhyMinLvl", "rpt.pml")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "Repeater-0:Mode-StateSettings", "rpt.mos")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "Repeater-0:UpdDirection", "rpt.upd")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "Repeater-0:Mode-DetectionBtn", "rpt.mob")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "Repeater-0:ManufacturerRst", "rpt.rst")
        return new

    if debug > 1:
        print("Info: No rpt feature in driver.")
    return None

def getsettings_dpwm(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="dpwm", DName="PWM Dim Mode", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "DimPwm-0")
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, new)
        ddfile.createFeaPropertyFromDDProperty(new, sub, "DimPwm-0:Enable", "dpwm.en")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "DimPwm-0:CurveType", "dpwm.type")
        return new

    if debug > 1:
        print("Info: No dpwm feature in driver.")
    return None

def getDataGenerationValue(formatElement):
    if "ActiveRef" in formatElement.attrib:
        if formatElement.attrib["ActiveRef"] == "true":
            return "1"
    return "0"

def createDataGenerationElement(structure, name, value):
    prop = et.SubElement(structure, "Property", Name=name, Api="0")
    form = et.SubElement(prop, "Format")
    et.SubElement(form, "Scale", Default=value, Min="0", Max="1")


def getsettings_luminfo(structure, switches=None, constants=None, filename=""):

    newfeature = et.Element("Feature", Name="inf", DName="Luminaire Info", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "Bank1")   # Luminaire Info feature based on Bank1 subdevice, e.g. in AM18522

    if sub is not None:

        instances = 1
        ###
        ### unless further notice: don't allow multiple instances in luminaire info feature!
        ###
        # if "MultipleInstance" in sub.attrib and sub.attrib["MultipleInstance"] == "true":
        #     tmp = constants.find(".//Constant[@Name='MaxInstances']").attrib["Value"]
        #     instances = int(tmp)
        #     newfeature.attrib["Instances"] = str(instances)

        newfeature.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, newfeature)

        content = sub.find("./Property[@PgmName='Bank1:VendorSpecificContent']")    # only if PMD data is there
        version = sub.find("./Property[@PgmName='Bank1:ContentFormatVersion']/Format/Scale")    # only if PMD data is there

        if content is not None:
            if "PermW" in content.attrib:
                perm = content.attrib["PermW"]
            else:           # if PermW attribute is missing, add "always"
                perm = "7"

            lenght_raw = content.find("./Format/HexBytes")
            length = lenght_raw.attrib["Length"]

            ver = et.SubElement(newfeature, "Property",
                                DName="Content Format ID",
                                Name="inf.ver",
                                PermW=perm,
                                Map="Bank1:ContentFormatVersion",
                                Api="1", 
                                Unit="")

            form = et.SubElement(ver, "Format")
            for index in range(instances):

                enum = et.SubElement(form, "EnumValues", Default=version.attrib["Default"], Type="xsd:unsignedInt")
                if instances > 1:
                    enum.attrib["Instance"] = str(index)

                et.SubElement(enum, "EnumValue", Value="0", Label="Type 0")
                if int(length) < 42:    # if MB1 of driver is too small, driver does not support Luminaire Info feature
                    return None         # applies for some older drivers
                if int(length) >= 42:   # register that type 2 would fit
                    et.SubElement(enum, "EnumValue", Value="2", Label="Type 2")
                if int(length) >= 101:  # also type 3 will fit
                    et.SubElement(enum, "EnumValue", Value="3", Label="Type 3")
                if int(length) >= 125:  # also type 4 will fit
                    et.SubElement(enum, "EnumValue", Value="4", Label="Type 4")
                if int(length) >= 159:  # also type 5 will fit
                    et.SubElement(enum, "EnumValue", Value="5", Label="Type 5")
                                        # None is always possible
                et.SubElement(enum, "EnumValue", Value="65535", Label="None")

            cont = sub.find("./Property[@PgmName='Bank1:VendorSpecificContent']")
            details = et.SubElement(newfeature, "Property",
                                    DName="Content",
                                    Name="inf.con",
                                    PermW=perm,
                                    Map="Bank1:VendorSpecificContent",
                                    Api="1", 
                                    Unit="")
            if perm is not None:
                details.attrib["PermW"] = perm
            formatElement = cont.find("./Format")
            details.append(copy_format_element(formatElement, instances))
            useProgrammingDateValue = getDataGenerationValue(formatElement)

            cont = sub.find("./Property[@PgmName='Bank1:OEMGTIN']")
            details = et.SubElement(newfeature, "Property",
                                    DName="Luminaire Manufacturer GTIN",
                                    Name="inf.oemg",
                                    PermW=perm,
                                    Map="Bank1:OEMGTIN",
                                    Api="1", 
                                    Unit="")
            details.append(copy_format_element(cont.find("./Format"), instances))

            cont = sub.find("./Property[@PgmName='Bank1:OEMIdentification']")
            details = et.SubElement(newfeature, "Property", DName="Luminaire Identification Number", Name="inf.oemid", 
                                    PermW=perm, Map="Bank1:OEMIdentification", Api="1", Unit="")
            formatElement = cont.find("./Format")
            details.append(copy_format_element(formatElement, instances))
            generateLuminaireIdValue = getDataGenerationValue(formatElement)

            bank207 = getsubdevice(structure, "Bank207")
            if bank207 is not None:
                ebl = et.SubElement(newfeature, "Property", DName="Expected Ballast LifeTime", Name="inf.ebl",
                                    PermW=perm, Map="Bank207:ExpectedBallastLifeTime", Api="1", Unit="kh")
                ebl.append(copy_format_element(bank207.find("./Property[@PgmName='Bank207:ExpectedBallastLifeTime']/Format"), instances))

                idt = et.SubElement(newfeature, "Property", DName="Internal Driver Reference Temperature", Name="inf.idt",
                                    PermW=perm, Map="Bank207:IntDriverRefTemp", Api="1", Unit="°C")
                idt.append(copy_format_element(bank207.find("./Property[@PgmName='Bank207:IntDriverRefTemp']/Format"), instances))

                elt = et.SubElement(newfeature, "Property", DName="Expected LightSource LifeTime", Name="inf.elt",
                                    PermW=perm, Map="Bank207:ExpectedLightSourceLifeTime", Api="1", Unit="kh")
                elt.append(copy_format_element(bank207.find("./Property[@PgmName='Bank207:ExpectedLightSourceLifeTime']/Format"), instances))

            # add hidden data generation properties if necessary
            if generateLuminaireIdValue == "1" or useProgrammingDateValue == "1":
                createDataGenerationElement(newfeature, "inf.GenerateLuminaireId", generateLuminaireIdValue)
                createDataGenerationElement(newfeature, "inf.UseProgrammingDate", useProgrammingDateValue)
            return newfeature
        else:
            if debug > 1:
                print("no VendorSpecificContent in Bank1")

    if debug > 1:
        print("Info: No LumInfo feature in driver.")
    return None


def getsettings_ledmodule(structure, switches=None, constants=None, filename=""):
    # this is a virtual feature, not directly related to a subdevice
    # so create from scratch

    newfeature = et.Element("Feature", 
                             Name="led", 
                             DName="LED Module Operating Data", 
                             Mapping="29", Instances="1", 
                             Programming="true",
                             LockProgramming="false")

    sub = structure.find(".//LEDModule")        # LEDModule feature included, e.g. in AM28017

    if sub is not None:   # there is a LED Module required for this driver!

        twa = getsubdevice(structure, "TWA-0")
        if twa is not None:

            tmp = twa.find("./Property[@PgmName='TWA-0:LEDModuleID']/Format/Scale")
            id = int(tmp.attrib["Default"])

            if id == 0:   # check if empty (no led module assigned yet) => generate a new default parameter set
                        
                twa_mapped = {
                    "TWA-0:Tmod_Ch0":    ("led.tc", "Module Temperature - Cold White", "°C"),
                    "TWA-0:Tmod_Ch1":    ("led.tw", "Module Temperature - Warm White", "°C"),
                    "TWA-0:Ta":          ("led.ta", "Ambient Temperature", "°C"),
                    "TWA-0:Flux-Led0":   ("led.cf", "Cold White - Luminous Flux", "lm"),
                    "TWA-0:Flux-Led1":   ("led.wf", "Warm White - Luminous Flux", "lm"),
                }

                others = [
                    ("led.id", "ModuleID", "", "0", "0", "281474976710655", "1"), 
                    ("led.mo", "Mode", "", "2", "0", "2", "1"),
                    ("led.cx", "Cold White - Color Coordinate x", "", "3095", "2200", "6000", "0.0001"),
                    ("led.cy", "Cold White - Color Coordinate y", "", "3231", "2200", "5000", "0.0001"),
                    ("led.wx", "Warm White - Color Coordinate x", "", "4571", "2200", "6000", "0.0001"),
                    ("led.wy", "Warm White - Color Coordinate y", "", "4077", "2200", "5000", "0.0001"),
                    ("led.ct", "Cold White - Color Temperature", "K", "6000", "0", "10000", "1"),
                    ("led.wt", "Warm White - Color Temperature", "K", "2700", "0", "10000", "1")]

                for elem in twa_mapped:
                    details = getproperty(twa, elem)
                    en = et.SubElement(newfeature, "Property", Api="1", Name=twa_mapped[elem][0], DName=twa_mapped[elem][1], 
                                       Map=elem, Unit=twa_mapped[elem][2])
                    en.append(copy_format_element(details.find("./Format"), 1))
                    if "PermW" in details.attrib:
                        en.attrib["PermW"] = details.attrib["PermW"]

                for elem in others:
                    pro = et.SubElement(newfeature, "Property", Api="1", Name=elem[0], DName=elem[1], Map="", Unit=elem[2])
                    sc = et.SubElement(pro, "Format")
                    fo = et.SubElement(sc, "Scale", Default=elem[3], Min=elem[4], Max=elem[5], Multiplier=elem[6])

                # to do: get permissions from TWA-0 subdevice and write into fea %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

                return newfeature

            else:         

                twa_mapped = {
                    "TWA-0:LEDModuleID": ("led.id", "ModuleID", ""),
                    "TWA-0:Tmod_Ch0":    ("led.tc", "Module Temperature - Cold White", "°C"),
                    "TWA-0:Tmod_Ch1":    ("led.tw", "Module Temperature - Warm White", "°C"),
                    "TWA-0:Ta":          ("led.ta", "Ambient Temperature", "°C"),
                    "TWA-0:Flux-Led0":   ("led.cf", "Cold White - Luminous Flux", "lm"),
                    "TWA-0:Flux-Led1":   ("led.wf", "Warm White - Luminous Flux", "lm"),
                }

                for elem in twa_mapped:
                    details = getproperty(twa, elem)
                    en = et.SubElement(newfeature, "Property", Api="1", Name=twa_mapped[elem][0], DName=twa_mapped[elem][1], 
                                       Map=elem, Unit=twa_mapped[elem][2])
                    en.append(copy_format_element(details.find("./Format"), 1))
                    if "PermW" in details.attrib:
                        en.attrib["PermW"] = details.attrib["PermW"]

                # "correct" an old TW Module ID: the modules with ID 15 are no longer in the LEDmodulestore.
                # but they are the same as ID11, thus change the ID to 11 in imported file.
                if id == 15:
                    newfeature.find("./Property[@Name='led.id']/Format/Scale").attrib["Default"]="11"

                # correct the range for property "module ID" (to allow extended module ID)
                idtarget = newfeature.find("./Property[@Name='led.id']/Format/Scale")
                idtarget.attrib["Max"] = "281474976710655"

                if id==65535:  # "correct" id if present LED module is a new module with "extended Module ID" 
                    tmp = twa.find("./Property[@PgmName='TWA-0:LEDModuleIDExt']/Format/Scale")
                    extidsource = tmp.attrib["Default"] if tmp is not None else '64000'
                    extidtarget = newfeature.find("./Property[@Name='led.id']/Format/Scale")
                    extidtarget.attrib["Default"] = extidsource

                twa_transformed = {
                    "TWA-0:X-Led0": ("led.cx", "Cold White - Color Coordinate x", "", "xsd:floatSingle"),
                    "TWA-0:Y-Led0": ("led.cy", "Cold White - Color Coordinate y", "", "xsd:floatSingle"),
                    "TWA-0:X-Led1": ("led.wx", "Warm White - Color Coordinate x", "", "xsd:floatSingle"),
                    "TWA-0:Y-Led1": ("led.wy", "Warm White - Color Coordinate y", "", "xsd:floatSingle")
                }

                for elem in twa_transformed:
                    details = getproperty(twa, elem)
                    en = et.SubElement(newfeature, "Property", Api="1", Name=twa_transformed[elem][0], DName=twa_transformed[elem][1], 
                                       Map="", Unit=twa_transformed[elem][2])
                    sc = et.SubElement(en, "Format")
                    form = details.find(".//Scale")
                    fo = et.SubElement(sc, "Scale", Default=str(int(float(form.attrib["Default"])*10000.0)), 
                                                    Min=str(int(float(form.attrib["Min"])*10000.0)), 
                                                    Max=str(int(float(form.attrib["Max"])*10000.0)), 
                                                    Multiplier="0.0001")
                    if "PermW" in details.attrib:
                        en.attrib["PermW"] = details.attrib["PermW"]
                
                x = float(twa.find(".//Property[@PgmName='TWA-0:X-Led0']/Format/Scale").attrib["Default"])
                y = float(twa.find(".//Property[@PgmName='TWA-0:Y-Led0']/Format/Scale").attrib["Default"])

                en = et.SubElement(newfeature, "Property", Api="1", DName="Cold White - Color Temperature",  
                                    Unit="K", Map="", Name="led.ct")
                fo = et.SubElement(en, "Format")
                sc = et.SubElement(fo, "Scale")
                sc.attrib["Default"] = str(xy_to_kelvin(x, y))
                sc.attrib["Min"] = "1"
                sc.attrib["Max"] = "10000"
                sc.attrib["Multiplier"] = "1"

                x = float(twa.find(".//Property[@PgmName='TWA-0:X-Led1']/Format/Scale").attrib["Default"])
                y = float(twa.find(".//Property[@PgmName='TWA-0:Y-Led1']/Format/Scale").attrib["Default"])

                en = et.SubElement(newfeature, "Property", Api="1", DName="Warm White - Color Temperature", 
                                    Unit="K", Map="", Name="led.wt")
                fo = et.SubElement(en, "Format")
                sc = et.SubElement(fo, "Scale")
                sc.attrib["Default"] = str(xy_to_kelvin(x, y))
                sc.attrib["Min"] = "1"
                sc.attrib["Max"] = "10000"
                sc.attrib["Multiplier"] = "1"

                # finally handle the LED Module mode      
                mode = "2"  # default is premium
                if "Type" in sub.attrib:                # in the import case, this attribute exists
                    tmp = sub.attrib["Type"].lower()    # then get the LED Module Type from LED Module entry
                    if tmp == "p": mode = "2"
                    if tmp == "a": mode = "1"
                    if tmp == "b": mode = "0"
                else:                                   # in the raw import, this attribute does not exist
                                                        # then "estimate" the mode from TWA-enable
                                                        # caution: advanced and premium cannot be distinguished!
                    mo = twa.find(".//Property[@PgmName='TWA-0:Enable']/Format/Scale").attrib["Default"]
                    if mo == "0": mode = "0"
                    if mo == "6": mode = "2"
        
                en = et.SubElement(newfeature, 
                                    "Property", 
                                    Api="1", 
                                    DName="Mode", 
                                    Unit="", 
                                    Map="", 
                                    Name="led.mo")
                fo = et.SubElement(en, "Format")
                sc = et.SubElement(fo, "Scale")
                sc.attrib["Default"] = mode
                sc.attrib["Min"] = "0"
                sc.attrib["Max"] = "2"

                return newfeature 

        else:
            print("found LED Module without TWA Subdevice --> error")
            return

    if debug > 1:
        print("Info: No LED module feature in driver.")
    return None


def getsettings_md(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="md", DName="MainsDIM", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "MD-0")   # MainDim feature based on MD-0
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)

        # copy the <scale> properties: there is one dd file bug: name of MD:StopLevel is not consistent to property naming scheme
        md0_scale_props = {
            "MD-0:StartVoltage": ("md.sv", "Start Voltage", "V"),
            "MD-0:StartLevel": ("md.sl", "StartLevel", "%"),
            "MD-0:StopVoltage": ("md.ev", "Stop Voltage", "V"),
            "MD:StopLevel": ("md.el", "Stop Level", "%"), }

        for elem in md0_scale_props:
            details = getproperty(sub, elem)
            en = et.SubElement(new, "Property", DName=md0_scale_props[elem][1], Name=md0_scale_props[elem][0], 
                               Map=elem, Api="1", Unit=md0_scale_props[elem][2])
            en.append(copy_format_element(details.find("./Format"), 1))
            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

        # get the switches of MD-0 feature
        details = getproperty(sub, "MD-0:Control-MDEnable")
        tmp = details.find("./Format/Scale")
        if tmp is not None:
            val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="MD-0:Control-MDEnable", Map="MD-0:Control-MDEnable", SetValue=val)
        return new

    if debug > 1:
        print("Info: no pd feature in the driver.")
    return None


def getsettings_opstime(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="ops", DName="Operating Time", Instances="1")

    it = constants.find(".//Constant[@Name='NfcType']")
    if it is not None:
        nfcversion = it.attrib["Value"]
    else:
        nfcversion = "0"

    if nfcversion == "3":
        new.attrib["Mapping"] = "25"    # in this case a differnt mapping has to be applied (adaptation to fixed x-times)
        nfc3 = True
    else:
        new.attrib["Mapping"] = "0"     # all other cases
        nfc3 = False

    sub = getsubdevice(structure, "Info-0")   # Operating Time feature based on Info-0 subdevice, e.g. in AM18316
    if sub is not None:

        instances = getFeatureInstanceCount(sub, new, constants)

        new.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, new)

        prop = sub.find("./Property[@PgmName='Info-0:LampOperationCounter']")  # this is mandatory prop in Info-0, so dont check if this exists

        ten = et.SubElement(new, "Property", DName="Enable", Name="ops.en", Map="", Api="1", Unit="")
        form1 = et.SubElement(ten, "Format")

        for index in range(instances):
            tmp = et.SubElement(form1, "Scale", Default="1", Min="1", Max="1", Type="xsd:boolean")   # ops time is always enabled and cannot be disabled
            if instances > 1:
                tmp.attrib["Instance"] = str(index)

        # if nfc3 == True:
        #     tti = et.SubElement(new, "Property", DName="Time", Name="ops.ti", Map="", Api="1", Unit="kh")
        # else:
        #     tti = et.SubElement(new, "Property", DName="Time", Name="ops.ti", Map="Info-0:LampOperationCounter", Api="1", Unit="kh")
        tti = et.SubElement(new, "Property", DName="Time", Name="ops.ti", Map="Info-0:LampOperationCounter", Api="1", Unit="kh")
        form = et.SubElement(tti, "Format")

        sources = prop.findall("./Format/Scale")
        index = 0
        for src in sources:
            tmp = et.SubElement(form, "Scale",
                                Default=src.attrib["Default"],
                                Min=src.attrib["Min"],
                                Max=src.attrib["Max"],
                                Multiplier=str(0.0000166666666)     # the programming value is in min, the API value shall be in kh, i.e. divide by 60000
                                )
            if instances > 1:
                tmp.attrib["Instance"] = str(index)
            index += 1

        if "PermW" in prop.attrib:
            ten.attrib["PermW"] = prop.attrib["PermW"]
            tti.attrib["PermW"] = prop.attrib["PermW"]

        return new

    if debug > 1:
        print("Info: No Ops Time feature in driver.")
    return None


def getsettings_pd(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="pd", DName="Presence Detection", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "PD-0")   # is presence detection feature based on PD-0 subdevice in this driver?
    if sub is not None:
        instances = getFeatureInstanceCount(sub, new, constants)
        
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)

        # copy the <scale> properties
        pd0_scale_props = {
            "PD-0:PDLevel": ("pd.l", "PD Level", "%"),
            # "PD-0:StartFadeTime":("pd.sf","Start Fade Time"),
            # "PD-0:EndFadeTime":("pd.ef","End Fade Time"),
            # "PD-0:HoldTime":("pd.h","Hold Time")
        }

        for elem in pd0_scale_props:
            details = getproperty(sub, elem)
            en = et.SubElement(new, "Property",
                               DName=pd0_scale_props[elem][1],
                               Name=pd0_scale_props[elem][0],
                               Map=elem,
                               Api="1", 
                               Unit=pd0_scale_props[elem][2])
            en.append(copy_format_element(details.find("./Format"), instances))
            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

        createFeaRefEnumPropFromScale(sub, new, "PD-0:StartFadeTime", "pd.sf", "Start Fade Time", "s", "0", "240", "astrodimfade", instances)
        createFeaRefEnumPropFromScale(sub, new, "PD-0:HoldTime", "pd.h", "Hold Time", "s", "0", "240", "astroswitchfade", instances)
        createFeaRefEnumPropFromScale(sub, new, "PD-0:EndFadeTime", "pd.ef", "End Fade Time", "s", "0", "240", "astrodimfade", instances)

        # get the switches of PD-0 feature
        val = getMultiInstancePropertyValues(sub, "PD-0:Control-PDEnable")
        et.SubElement(switches, "Switch", Name="PD-0:Control-PDEnable", Map="PD-0:Control-PDEnable", SetValue=val)
        return new

    if debug > 1:
        print("Info: no pd feature in the driver.")
    return None


def getsettings_sd(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="sd", DName="StepDIM/LSI", Mapping="0", Instances="1")

    sub = getsubdevice(structure, "SD-0")   # stepdim feature based on SD-0
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)

        # copy the <scale> properties
        sd0_scale_props = {
            "SD-0:SDLevel": ("sd.l1", "Dim Level", "%"),
            "SD-0:Nominallevel": ("sd.l0", "Nominal Level", "%"),
        }
        for elem in sd0_scale_props:
            details = getproperty(sub, elem)
            en = et.SubElement(new, "Property",
                               DName=sd0_scale_props[elem][1],
                               Name=sd0_scale_props[elem][0],
                               Map=elem,
                               Api="1",
                               Unit=sd0_scale_props[elem][2])
            en.append(copy_format_element(details.find("./Format"), 1))
            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

        # handle enum properties: use default from dd but correct other values
        sdf0 = sub.find(".//Property[@PgmName='SD-0:StartupFadeTime']/Format/Scale").attrib["Default"]
        create_enum_property(new, "Switch on Fade Time", "sd.f0", "SD-0:StartupFadeTime", None, "astroswitchfade", "0", "240", sdf0)
        sdf1 = sub.find(".//Property[@PgmName='SD-0:StartFadeTime']/Format/Scale").attrib["Default"]
        create_enum_property(new, "Start Fade Time", "sd.f1", "SD-0:StartFadeTime", None, "astrodimfade", "0", "240", sdf1)
        sdh1 = sub.find(".//Property[@PgmName='SD-0:HoldTime']/Format/Scale").attrib["Default"]
        create_enum_property(new, "Hold Time", "sd.h1", "SD-0:HoldTime", None, "astroswitchfade", "0", "240", sdh1)
        sdf2 = sub.find(".//Property[@PgmName='SD-0:EndFadeTime']/Format/Scale").attrib["Default"]
        create_enum_property(new, "End Fade Time", "sd.f2", "SD-0:EndFadeTime", None, "astrodimfade", "0", "240", sdf2)

        # get the switches of SD-0 feature
        details = getproperty(sub, "SD-0:Control-SDEnable")
        tmp = details.find("./Format/Scale")
        if tmp is not None:
            val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="SD-0:Control-SDEnable", Map="SD-0:Control-SDEnable", SetValue=val)
        return new

    sub = getsubdevice(structure, "SD-1")   # stepdim feature based on SD-1
    if sub is not None:
        instances = getFeatureInstanceCount(sub, new, constants)

        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)

        # copy the <scale> properties
        sd1_scale_props = {
            "SD-1:SDLevel": ("sd.l1", "Dim Level", "%"),
            "SD-1:Nominallevel": ("sd.l0", "Nominal Level", "%"),
        }
        for elem in sd1_scale_props:
            details = getproperty(sub, elem)
            en = et.SubElement(new, "Property",
                               DName=sd1_scale_props[elem][1],
                               Name=sd1_scale_props[elem][0],
                               Map=elem,
                               Api="1",
                               Unit=sd1_scale_props[elem][2])
            en.append(copy_format_element(details.find("./Format"), instances))
            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

        # handle enum properties: use default from dd but correct other values
        createFeaRefEnumPropFromScale(sub, new, "SD-1:StartupFadeTime", "sd.f0", "Switch on Fade Time", "s", "0", "3600", "astroswitchfade16b", instances)
        createFeaRefEnumPropFromScale(sub, new, "SD-1:StartFadeTime", "sd.f1", "Start Fade Time", "s", "0", "240", "astrodimfade", instances)
        createFeaRefEnumPropFromScale(sub, new, "SD-1:HoldTime", "sd.h1", "Hold Time", "s", "0", "240", "astroswitchfade", instances)
        createFeaRefEnumPropFromScale(sub, new, "SD-1:EndFadeTime", "sd.f2", "End Fade Time", "s", "0", "240", "astrodimfade", instances)

        # get the switches of SD-1 feature
        val = getMultiInstancePropertyValues(sub, "SD-1:Control-SDEnable")
        et.SubElement(switches, "Switch", Name="SD-1:Control-SDEnable", Map="SD-1:Control-SDEnable", SetValue=val)
        return new

    if debug > 1:
        print("Info: no sd feature in the driver.")
    return None


def getsettings_sso(structure, switches=None, constants=None, filename=""):
    new = et.Element("Feature", Name="sso", DName="Soft Switch Off", Instances="1")

    cur = getsubdevice(structure, "OTConfig-1")
    if cur is not None:
        sso = cur.find("./Property[@PgmName='OTConfig-1:ModeSetting-SoftSwitchOff']")   # sso feature based on OTConfig-1, e.g. in AB32361
        if sso is not None:
            new.attrib["Programming"] = cur.attrib["Programming"]       
            copy_LockProgramming(cur, new)

            new.attrib["Mapping"] = "15"
            ssoset = sso.find("./Format/Scale").attrib["Default"]       # for this MPC, the SSO properties (en & ti) are combined in one property 
            ssoen = "0" if ssoset == "0" else "1"                       # deflate them

            tmp = et.SubElement(new, "Property", DName="Enable", Name="sso.en", Map="", Api="1", Unit="")
            form = et.SubElement(tmp, "Format")
            et.SubElement(form, "Scale", Default=ssoen, Min="0", Max="1", Type="xsd:boolean")

            tmp = et.SubElement(new, "Property", DName="Time", Name="sso.ti", Map="", Api="1", Unit="s")
            form = et.SubElement(tmp, "Format")
            et.SubElement(form, "EnumValues", Default=ssoset, Min="1", Max="7", Off="0", Ref="tdfade")

            # potentially add dummy "SSOMode-Enable" switch for this case to harmonize the bahavior between different drivers

        else:  # no sso availabe despite OTConfig-1, i.e. clear "cur" so that search for SSO can be continued
            cur = None

    if cur is None:
        cur = getsubdevice(structure, "GFM-0")   # sso feature based on GFM-0 subdevice, e.g. in AM12413
        if cur is not None:
            sso = cur.find("./Property[@PgmName='GFM-0:SSOMode-Enable']")
            if sso is not None:
                new.attrib["Mapping"] = "0"
                new.attrib["Programming"] = cur.attrib["Programming"]       # copy "programming" attribute from source
                copy_LockProgramming(cur, new)

                ssoen = sso.find("./Format/Scale").attrib["Default"]
                tmp = et.SubElement(new, "Property", DName="Enable", Name="sso.en", Map="GFM-0:SSOMode-Enable", Api="1", Unit="")
                form = et.SubElement(tmp, "Format")
                et.SubElement(form, "Scale", Default=ssoen, Min="0", Max="1", Type="xsd:boolean")

                ssoset = cur.find("./Property[@PgmName='GFM-0:SSOMode-Time']/Format/Scale").attrib["Default"]
                tmp = et.SubElement(new, "Property", DName="Time", Name="sso.ti", Map="GFM-0:SSOMode-Time", Api="1", Unit="s")
                form = et.SubElement(tmp, "Format")
                et.SubElement(form, "EnumValues", Default=ssoset, Min="1", Max="7", Off="0", Ref="tdfade")

                if "PermW" in cur.attrib:
                    new.attrib["PermW"] = cur.attrib["PermW"]

                # add switches for SSO
                allowed = cur.find("./Property[@PgmName='GFM-0:SSOMode-Allowed']/Format/Scale")
                if allowed is not None:
                    val = allowed.attrib["Default"]
                    et.SubElement(switches, "Switch", Name="GFM-0:SSOMode-Allowed", Map="GFM-0:SSOMode-Allowed", SetValue=val)
                else:
                    print("*** error: missing GFM-0:SSOMode-Allowed!")

            else:  # no sso feature present despite GFM-0 subdevice, i.e. clear "cur" so that search is continued
                cur = None

    if cur is None:
        cur = getsubdevice(structure, "GFM-1")   # sso feature based on GFM-1 subdevice, e.g. AM18316
        if cur is not None:

            sso = cur.find("./Property[@PgmName='GFM-1:SSOMode-Enable']")

            if sso is not None:
                new.attrib["Mapping"] = "0"
                new.attrib["Programming"] = cur.attrib["Programming"]       # copy "programming" attribute from source
                copy_LockProgramming(cur, new)

                ssomax = sso.find("./Format/Scale").attrib["Max"]
                if ssomax=='0':                         # sso feature cannot be enabled ?
                    cur = None                          # no -> leave sso feature creation here
                else:
                    ssoen = sso.find("./Format/Scale").attrib["Default"]
                    tmp = et.SubElement(new, "Property", DName="Enable", Name="sso.en", Map="GFM-1:SSOMode-Enable", Api="1", Unit="")
                    form = et.SubElement(tmp, "Format")
                    et.SubElement(form, "Scale", Default=ssoen, Min="0", Max="1", Type="xsd:boolean")

                    ssoset = cur.find("./Property[@PgmName='GFM-1:SSOMode-Time']/Format/Scale").attrib["Default"]
                    tmp = et.SubElement(new, "Property", DName="Time", Name="sso.ti", Map="GFM-1:SSOMode-Time", Api="1", Unit="s")
                    form = et.SubElement(tmp, "Format")
                    et.SubElement(form, "EnumValues", Default=ssoset, Min="1", Max="7", Off="0", Ref="tdfade")

                    if "PermW" in cur.attrib:
                        new.attrib["PermW"] = cur.attrib["PermW"]

                    # add switches for SSO
                    allowed = cur.find("./Property[@PgmName='GFM-1:SSOMode-Allowed']/Format/Scale")
                    if allowed is not None:
                        val = allowed.attrib["Default"]
                        et.SubElement(switches, "Switch", Name="GFM-1:SSOMode-Allowed", Map="GFM-1:SSOMode-Allowed", SetValue=val)
                    else:
                        print("*** error: missing GFM-1:SSOMode-Allowed despite SSO")

            else:  # no sso feature despite GFM-1, i.e. clear "cur" so that search is continued
                cur = None

    if cur is not None:
        return new

    if debug > 1:
        print("Info: No sso feature in the driver.")
    return None


def getsettings_tdcf(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="td", DName="TouchDIM", Instances="1")
    tmp = et.SubElement(new, "Property", Api="1", Name="td.mo", DName="TD Mode", Map="", Unit="")
    tmp = et.SubElement(tmp, "Format")
    tmp = et.SubElement(tmp, "Scale", Default="1", Min="0", Max="3")

    # add mains frequency property
    mf = structure.find(".//Property[@PgmName='MainsFrequency']/Format/Scale")
    if mf is not None:
        value = mf.attrib["Default"]
        mf = createFeaProperty('td.mf', 'Mains Frequency', 'MainsFrequency', 'Hz', 'EnumValues', 1, Default=value)
        enums = mf.find(".//Format/EnumValues")
        et.SubElement(enums, 'EnumValue', Value="50", Label="50")
        et.SubElement(enums, 'EnumValue', Value="60", Label="60")
        new.append(mf)

    # check for dependent features of TD:
    dependent = detectDependencies(structure, "td", True)
    new.attrib["Dep"] = dependent

    sub = getsubdevice(structure, "TDiCorridor-0")   # Touchdim feature based on TDiCorridor-0, e.g. aa67888.xml
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)
        new.attrib["Mapping"] = "9"

        tdcf0_props = [
            ("TDiCorridor-0:TDConfig-ShortPushEnable", "td.esp", "Enable Short-Push", "scale", ""),
            ("TDiCorridor-0:TDConfig-DoublPushEnable", "td.edp", "Enable Double-Push", "scale", ""),
            ("TDiCorridor-0:TDConfig-LongPushEnable", "td.elp", "Enable Long-Push", "scale", ""),
            ("TDiCorridor-0:TDConfig-PSEnable", "td.epd", "Enable Presence Detection", "scale", ""),
            ("TDiCorridor-0:TDConfig-LSEnable", "td.els", "Enable Light Sensor", "scale", ""),
            ("TDiCorridor-0:TDConfig-AutoDisPSEnable", "td.eap", "Auto Disable PD", "scale", ""),
            ("TDiCorridor-0:TDConfig-HolidayEnable", "td.ehm", "Enable Holiday Mode", "scale", ""),
            ("TDiCorridor-0:TDPowerOnLevel", "td.po", "Mains Power On Level", "scale", ""),
            ("TDiCorridor-0:TDSwitchOnLevel", "td.so", "TD Switch on Level", "scale", ""),
            ("TDiCorridor-0:FadeUpTime", "td.fu", "TD Fade Time [1] / CF Fade Time [1]", "tdfade", "s"),
            ("TDiCorridor-0:FadeDownTime", "td.fd", "TD Fade Time [4] / CF Fade Time [3]", "tdfade", "s"),
            ("TDiCorridor-0:PSTimeout", "td.on", "TD Hold Time [2]", "tdh1", "s"),
            ("TDiCorridor-0:TDStandByTime", "td.sb", "TD Hold Time [5]", "tdh1", "s"),
            # ("TDiCorridor-0:CFHoldTime0","td.h0", "CF Hold Time [2]","scale"),    # is an exception with own mapping
            ("TDiCorridor-0:CFHoldTime1", "td.h1", "CF Hold Time [4]", "tdh3", "s"),
            ("TDiCorridor-0:CFHoldTime2", "td.h2", "CF Hold Time [6]", "tdh3", "s"),
            ("TDiCorridor-0:CFFadeTime2", "td.fs", "TD Fade Time [3] / CF Fade Time [3]", "tdfade", "s"),
            ("TDiCorridor-0:CFLevel0", "td.l0", "Corridor Operative Level (Level 0)", "scale", ""),
            ("TDiCorridor-0:CFLevel1", "td.l1", "Corridor Standby Level 1", "scale", ""),
            ("TDiCorridor-0:CFLevel2", "td.l2", "Corridor Standby Level 2", "scale", ""),
        ]

        # copy all tdcf0 <scale> properties
        for elem in tdcf0_props:

            details = getproperty(sub, elem[0])
            en = et.SubElement(new, "Property",
                               DName=elem[2],
                               Name=elem[1],
                               Map=elem[0],
                               Api="1",
                               Unit=elem[4])

            if elem[3] == "scale":

                tmp = details.find("./Format/Scale")
                if elem[0] == "TDiCorridor-0:TDPowerOnLevel":  # fix import for some old dd files: correct the source before copy
                    tmp.attrib["Max"] = "254"
                    tmp.attrib["Off"] = "255"
                    tmp.attrib["Initial"] = "254"
                # elif elem[0] == "TDiCorridor-0:TDSwitchOnLevel":
                #     tmp.attrib["Initial"] = "254"

                en.append(copy_format_element(details.find("./Format"), 1))

            else:
                src = details.find("./Format/Scale")
                form = et.SubElement(en, "Format")
                enum = et.SubElement(form, "EnumValues",
                                     Ref=elem[3],
                                     Default=src.attrib["Default"],
                                     Min=src.attrib["Min"],
                                     Max=src.attrib["Max"])

            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

        # handle the exceptions
        details = getproperty(sub, "TDiCorridor-0:CFHoldTime0")
        prop = et.SubElement(new, "Property", DName="CF Hold Time [2]", Name="td.h0", Map="", Api="1", Unit="s")
        tmp = details.find("./Format/Scale")

        # there is an error in most old dd files: the is CFHoldTime0 must be <255, but most dd files have max=255
        # moreover, the value 255 in the generic TD model stands for "infinite". This must also not be used by drives with this MPC type.
        # therefore clip potentially affected attributes of this property (i.e. max and default) to 254!

        max = int(tmp.attrib["Max"])
        if max < 254:
            max = max + 1
        else:
            max = 254

        default = int(tmp.attrib["Default"])
        if default < 254:
            default = default + 1
        else:
            default = 254

        form = et.SubElement(prop, "Format")
        et.SubElement(form, "EnumValues",
                      Default=str(default),
                      Min=str(int(tmp.attrib["Min"]) + 1),
                      Max=str(max),
                      Ref="tdh3")

        if "PermW" in details.attrib:
            en.attrib["PermW"] = details.attrib["PermW"]

        # get the switches of the TDiCorridor-0 feature
        details = getproperty(sub, "TDiCorridor-0:TDConfig-TDCFEnable")
        tmp = details.find("./Format/Scale")
        if tmp is not None:
            val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="TDiCorridor-0:TDConfig-TDCFEnable", Map="TDiCorridor-0:TDConfig-TDCFEnable", SetValue=val)

        details = getproperty(sub, "TDiCorridor-0:TDControl-TDOperate")
        tmp = details.find("./Format/Scale")
        if tmp is not None:
            val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="TDiCorridor-0:TDControl-TDOperate", Map="TDiCorridor-0:TDControl-TDOperate", SetValue=val)

        details = getproperty(sub, "TDiCorridor-0:TDControl-CFOperate")
        tmp = details.find("./Format/Scale")
        if tmp is not None:
            val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="TDiCorridor-0:TDControl-CFOperate", Map="TDiCorridor-0:TDControl-CFOperate", SetValue=val)

        return new

    sub = getsubdevice(structure, "TDiCorridor-1")   # Touchdim feature based on TDiCorridor-1, e.g. ab36003.xml
    if sub is not None:
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)
        new.attrib["Mapping"] = "0"

        tdcf1_props = [
            # structure: (dd file property name, api name, api display name, format type to use: scale or enum list type to map)
            ("TDiCorridor-1:TDConfig-ShortPushEnable", "td.esp", "Enable Short-Push", "scale", ""),
            ("TDiCorridor-1:TDConfig-DoublPushEnable", "td.edp", "Enable Double-Push", "scale", ""),
            ("TDiCorridor-1:TDConfig-LongPushEnable", "td.elp", "Enable Long-Push", "scale", ""),
            ("TDiCorridor-1:TDConfig-PSEnable", "td.epd", "Enable Presence Detection", "scale", ""),
            ("TDiCorridor-1:TDConfig-LSEnable", "td.els", "Enable Light Sensor", "scale", ""),
            ("TDiCorridor-1:TDConfig-AutoDisPSEnable", "td.eap", "Auto Disable PD", "scale", ""),
            ("TDiCorridor-1:TDConfig-HolidayEnable", "td.ehm", "Enable Holiday Mode", "scale", ""),
            ("TDiCorridor-1:TDPowerOnLevel", "td.po", "TD Power On Level", "scale", ""),
            ("TDiCorridor-1:TDSwitchOnLevel", "td.so", "TD Switch on Level", "scale", ""),
            ("TDiCorridor-1:FadeUpTime", "td.fu", "TD Fade Time [1] / CF Fade Time [1]", "tdfade", "s"),
            ("TDiCorridor-1:FadeDownTime", "td.fd", "TD Fade Time [4] / CF Fade Time [3]", "tdfade", "s"),
            ("TDiCorridor-1:PSTimeout", "td.on", "TD Hold Time [2]", "tdh1", "s"),
            ("TDiCorridor-1:TDStandByTime", "td.sb", "TD Hold Time [5]", "tdh1", "s"),
            ("TDiCorridor-1:CFHoldTime0", "td.h0", "CF Hold Time [2]", "tdh4", "s"),   # T4TWEB-770: changed from tdh3 to tdh4 
            ("TDiCorridor-1:CFHoldTime1", "td.h1", "CF Hold Time [4]", "tdh3", "s"),
            ("TDiCorridor-1:CFHoldTime2", "td.h2", "CF Hold Time [6]", "tdh3", "s"),
            ("TDiCorridor-1:CFFadeTime2", "td.fs", "TD Fade Time [3] / CF Fade Time [5]", "tdfade", "s"),
            ("TDiCorridor-1:CFLevel0", "td.l0", "Corridor Operative Level (Level 0)", "scale", ""),
            ("TDiCorridor-1:CFLevel1", "td.l1", "Corridor Standby Level 1", "scale", ""),
            ("TDiCorridor-1:CFLevel2", "td.l2", "Corridor Standby Level 2", "scale", ""),
            ("TDiCorridor-1:CFPowerOnBehavior", "td.su", "Corridor Power On Behavior", "copy", ""),
        ]

        # copy all tdcf1 properties
        for elem in tdcf1_props:

            try:
                details = getproperty(sub, elem[0])

                en = et.SubElement(new, "Property",
                                DName=elem[2],
                                Name=elem[1],
                                Map=elem[0],
                                Api="1",
                                Unit=elem[4])

                if elem[3] == "scale":

                    tmp = details.find("./Format/Scale")
                    if elem[0] == "TDiCorridor-1:TDPowerOnLevel" or elem[0] == "TDiCorridor-1:TDSwitchOnLevel":  # import: fix some old dd files: correct source before copying
                        tmp.attrib["Max"] = "254"
                        tmp.attrib["Off"] = "255"
                        tmp.attrib["Initial"] = "254"

                    en.append(copy_format_element(details.find("./Format"), 1))

                elif elem[3]=="copy":
                    en.append(copy_format_element(details.find("./Format"), 1))

                else:
                    src = details.find("./Format/Scale")
                    form = et.SubElement(en, "Format")
                    enum = et.SubElement(form, "EnumValues",
                                        Ref=elem[3],
                                        Default=src.attrib["Default"],
                                        Min=src.attrib["Min"],
                                        Max=src.attrib["Max"])

                if "PermW" in details.attrib:
                    en.attrib["PermW"] = details.attrib["PermW"]

            except:
                pass

        # get the switches of the TDiCorridor-1 feature
        details = getproperty(sub, "TDiCorridor-1:TDConfig-TDCFEnable")
        tmp = details.find("./Format/Scale")
        if tmp is not None:
            val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="TDiCorridor-1:TDConfig-TDCFEnable", Map="TDiCorridor-1:TDConfig-TDCFEnable", SetValue=val)

        return new

    sub = getsubdevice(structure, "TDiCorridor-5")   # Touchdim feature based on TDiCorridor-5, used in TW drivers
    if sub is not None:
        new.attrib["Mapping"] = "28"  
        new.attrib["Programming"] = sub.attrib["Programming"]
        copy_LockProgramming(sub, new)

        tdcf5_props = [
            # structure: (dd file property name, api name, api display name, format type to use: scale or enum list type to map)
            ("TDiCorridor-5:TDConfig-ShortPushEnable", "td.es1", "Enable Short-Push CH1", "scale", ""),
            ("TDiCorridor-5:TDConfig-DoublPushEnable", "td.ed1", "Enable Double-Push CH1", "scale", ""),
            ("TDiCorridor-5:TDConfig-LongPushEnable", "td.el1", "Enable Long-Push CH1", "scale", ""),
            ("TDiCorridor-5:TDConfig-PSEnable", "td.epd", "Enable Presence Detection", "scale", ""),
            ("TDiCorridor-5:TDConfig-LSEnable", "td.els", "Enable Light Sensor", "scale", ""),
            ("TDiCorridor-5:TDConfig-AutoDisPSEnable", "td.eap", "Auto Disable PD", "scale", ""),
            ("TDiCorridor-5:TDConfig-HolidayEnable", "td.ehm", "Enable Holiday Mode", "scale", ""),
            ("TDiCorridor-5:TDPowerOnLevel", "td.po", "TD Power On Level", "scale", ""),
            ("TDiCorridor-5:TDSwitchOnLevel", "td.so", "TD Switch on Level", "scale", ""),
            ("TDiCorridor-5:FadeUpTime", "td.fu", "TD Fade Time [1] / CF Fade Time [1]", "tdfade", "s"),
            ("TDiCorridor-5:FadeDownTime", "td.fd", "TD Fade Time [4] / CF Fade Time [3]", "tdfade", "s"),
            ("TDiCorridor-5:PSTimeout", "td.on", "TD Hold Time [2]", "tdh1", "s"),
            ("TDiCorridor-5:TDStandByTime", "td.sb", "TD Hold Time [5]", "tdh1", "s"),
            ("TDiCorridor-5:CFHoldTime0", "td.h0", "CF Hold Time [2]", "tdh4", "s"),
            ("TDiCorridor-5:CFHoldTime1", "td.h1", "CF Hold Time [4]", "tdh3", "s"),
            ("TDiCorridor-5:CFHoldTime2", "td.h2", "CF Hold Time [6]", "tdh3", "s"),
            ("TDiCorridor-5:CFFadeTime2", "td.fs", "TD Fade Time [3] / CF Fade Time [5]", "tdfade", "s"),
            ("TDiCorridor-5:CFLevel0", "td.l0", "Corridor Operative Level (Level 0)", "scale", ""),
            ("TDiCorridor-5:CFLevel1", "td.l1", "Corridor Standby Level 1", "scale", ""),
            ("TDiCorridor-5:CFLevel2", "td.l2", "Corridor Standby Level 2", "scale", ""),
            ("TDiCorridor-5:Config2-ShortPushEnable", "td.es2", "Enable Short-Push CH2", "scale", ""),
            ("TDiCorridor-5:Config2-DoublePushEnable", "td.ed2", "Enable Double-Push CH2", "scale", ""),
            ("TDiCorridor-5:Config2-LongPushEnable", "td.el2", "Enable Long-Push CH2", "scale", ""),
            ("TDiCorridor-5:TD2PowerOnLevel", "td.po2", "TD Power On Level CH2", "scale", ""),
            ("TDiCorridor-5:TD2SwitchOnLevel", "td.so2", "TD Switch on Level CH2", "scale", ""),
            ("TDiCorridor-5:CFPowerOnBehavior", "td.su", "Corridor Power On Behavior", "copy", ""),
            ]
        tdcf5_cprops = [
            ("TDiCorridor-5:TD2PowerOnColour", "td.poc", "Power On Color Temperature", "scale", "K"),
            ("TDiCorridor-5:TD2SwitchOnColour", "td.soc", "Switch On Color Temperature", "scale", "K")]

        # copy all tdcf5 properties
        for elem in tdcf5_props:
            tmp = sub.find("./Property[@PgmName='%s']" % elem[0])
            if tmp is not None:
                details = getproperty(sub, elem[0])
                en = et.SubElement(new, "Property",
                                DName=elem[2],
                                Name=elem[1],
                                Map=elem[0],
                                Api="1",
                                Unit=elem[4])

                if elem[3] == "scale":
                    if (elem[0] == "TDiCorridor-5:TDPowerOnLevel" or
                            elem[0] == "TDiCorridor-5:TDPowerOnLevel"):  # import fix for some old dd files: correct source before copying
                        tmp = details.find("./Format/Scale")
                        tmp.attrib["Max"] = "254"
                        tmp.attrib["Off"] = "255"
                        tmp.attrib["Initial"] = "254"

                    en.append(copy_format_element(details.find("./Format"), 1))
                elif elem[3]=="copy":
                    en.append(copy_format_element(details.find("./Format"), 1))

                else:
                    src = details.find("./Format/Scale")
                    form = et.SubElement(en, "Format")
                    enum = et.SubElement(form, "EnumValues",
                                        Ref=elem[3],
                                        Default=src.attrib["Default"],
                                        Min=src.attrib["Min"],
                                        Max=src.attrib["Max"])
                if "PermW" in details.attrib:
                    en.attrib["PermW"] = details.attrib["PermW"]
            else:
                if "TDiCorridor-5:CFPowerOnBehavior" == elem[0]:
                    print("info: missing optional property %s in TD-5 Subdevice" % elem[0])
                else:
                    print("error: missing property %s in TD-5 Subdevice" % elem[0])

        for elem in tdcf5_cprops:   # handle the color properties
            tmp = sub.find("./Property[@PgmName='%s']" % elem[0])
            if tmp is not None:

                pro = et.SubElement(new, "Property", Name=elem[1], DName=elem[2], Map="", Api="1", Unit=elem[3])
                src = tmp.find("./Format/Scale")
                fo = et.SubElement(pro, "Format")
                defaultValue = src.attrib["Default"]
                if defaultValue != "65535":
                    defaultValue = mirek_to_kelvin(defaultValue)
                sc = et.SubElement(fo, "Scale",
                                   Default=defaultValue,
                                   Min=mirek_to_kelvin(src.attrib["Max"]),
                                   Max=mirek_to_kelvin(src.attrib["Min"]),
                                   Initial=mirek_to_kelvin(src.attrib["Initial"]),
                                   #Off=mirek_to_kelvin(src.attrib["Off"]))
                                   Off="65535")
            else:
                print("error: missing the mandatory property %s in TD-5 Subdevice" % elem[0])

        # get the switches of TDiCorridor-5 feature
        details = getproperty(sub, "TDiCorridor-5:TDConfig-TDCFEnable")
        tmp = details.find("./Format/Scale")
        if tmp is not None:
            val = tmp.attrib["Default"]
        et.SubElement(switches, "Switch", Name="TDiCorridor-5:TDConfig-TDCFEnable", Map="TDiCorridor-5:TDConfig-TDCFEnable", SetValue=val)

        return new

    if debug > 1:
        print("Info: no TouchDim feature in the driver.")
    return None


def getsettings_thermprot(structure, switches=None, constants=None, filename=""):

    new = et.Element("Feature", Name="tp", DName="Thermal Protection", Instances="1")

    # ThermProt-0 was never used, so no support is required
    # ThermProt-1 is used in 3DIM drivers only. As those are not supported (descoped) in T4T-Cloud,
    # don't handle that subdevice type here.

    sub = getsubdevice(structure, "ThermProt-2")   #  Protection based on ThermProt-2, e.g. in AA64275
    # this SubDev uses Ohm and °C and thus values can be directly
    # copied from fea to API and vice versa
    if sub is not None:
        dep = detectDependencies(structure, "tp", False, cur="OTConfig-2")
        if dep != "tp":
            new.attrib["Dep"] = dep
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)
        new.attrib["Mapping"] = "18"

        tp_scale_props = {
            "ThermProt-2:Command-TPMEnable": ("tp.en", "Enable", True, ""),
            "ThermProt-2:Command-TempMeasureEnable": ("tp.mo", "Mode", True, ""),
            "ThermProt-2:StartDeratingResistance": ("tp.sr", "Start Derating Resistance", True, "Ohm"),
            "ThermProt-2:EndDeratingResistance": ("tp.er", "End Derating Resistance", True, "Ohm"),
            "ThermProt-2:ShutOffResistance": ("tp.or", "ShutOff Resistance", True, "Ohm"),
            "ThermProt-2:FinalPowerReductionLevel": ("tp.dl", "Derating Level", True, "%"),
            "PowerReductionStartTemperature": ("tp.st", "Start Derating Temperature", True, "°C"),
            "PowerReductionEndTemperature": ("tp.et", "End Derating Temperature", True, "°C"),
            "ShutDownTemperature": ("tp.ot", "ShutOff Temperature", True, "°C"),

            "ThermProt-2:SensorLevel25": ("tp.25", "Sensor Level at 25 °C", False, ""),
            "ThermProt-2:SensorLevel40": ("tp.40", "Sensor Level at 40 °C", False, ""),
            "ThermProt-2:SensorLevel55": ("tp.55", "Sensor Level at 55 °C", False, ""),
            "ThermProt-2:SensorLevel70": ("tp.70", "Sensor Level at 70 °C", False, ""),
            "ThermProt-2:SensorLevel85": ("tp.85", "Sensor Level at 85 °C", False, ""),
            "ThermProt-2:SensorLevel100": ("tp.10", "Sensor Level at 100 °C", False, ""),
        }

        # copy all tp <scale> properties
        for elem in tp_scale_props:
            details = getproperty(sub, elem)
            en = et.SubElement(new, "Property",
                               DName=tp_scale_props[elem][1],
                               Name=tp_scale_props[elem][0],
                               Map=elem,
                               Api="1" if tp_scale_props[elem][2] else "0",
                               Unit=tp_scale_props[elem][3])
            en.append(copy_format_element(details.find("./Format"), 1))

            if "PermW" in details.attrib:
                en.attrib["PermW"] = details.attrib["PermW"]

            # detects sensor attributes and correct their Max attrib (dd file error!)
            # i.e. tp.25 to tp.10
            postfix = tp_scale_props[elem][0][-2:]
            if postfix.isdigit():
                scale = en.find("./Format/Scale")
                scale.attrib["Max"] = "65535"

        # handle the exceptions: add stub element for NTC sensor list when temperature based mode is available
        mode = sub.find(".//Property[@PgmName='ThermProt-2:Command-TempMeasureEnable']/Format/Scale")
        if int(mode.attrib["Max"]) > 0:
            
            sensor = sub.find(".//Property[@PgmName='SensorType']/Format/Scale").attrib["Default"]

            prop = et.SubElement(new, "Property", DName="Sensor Type", Name="tp.se", Map="SensorType", Api="1", Unit="")
            form = et.SubElement(prop, "Format")
            enum = et.SubElement(form, "EnumValues", Ref="sensorlist", Default=sensor)

        # no switches in the ThermProt-2 feature

        # finally handle HwResources
        hw_resouces = sub.findall("./HwResource")
        for hwr in hw_resouces:
            new.append(deepcopy(hwr))
        return new

    sub = getsubdevice(structure, "ThermProtA-0")     # ThermProtA-0 in 2DIM Drivers, e.g. in AA68626
    if sub is not None:

        def tpma0_lev_to_level(value):
            value = int(value)
            return str(round(value / 327.68))

        def tmpa0_res_to_resistor(value):
            value = int(value)
            return str(round(5.0 / (value / 6736739.0 - 0.000338)))

        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)
        new.attrib["Mapping"] = "19"

        details = sub.find(".//Property[@PgmName='ThermProtA-0:EnableNtc']/Format/Scale")
        prop = et.SubElement(new, "Property", DName="Enable", Name="tp.en", Map="ThermProtA-0:EnableNtc", Api="1", Unit="")
        form = et.SubElement(prop, "Format")
        et.SubElement(form, "Scale",
                      Default=details.attrib["Default"],
                      Min=details.attrib["Min"],
                      Max=details.attrib["Max"],
                      Type="xsd:boolean")

        tp_res_props = {
            "ThermProtA-0:StartDeratingResistance": ("tp.sr", "Start Derating Resistance", "Ohm"),
            "ThermProtA-0:EndDeratingResistance": ("tp.er", "End Derating Resistance", "Ohm"),
        }

        # handle all tpa-0 "resistor" properties
        for elem in tp_res_props:
            details = sub.find(".//Property[@PgmName='%s']/Format/Scale" % elem)

            prop = et.SubElement(new, "Property",
                                 DName=tp_res_props[elem][1],
                                 Name=tp_res_props[elem][0],
                                 Map="",
                                 Api="1",
                                 Unit=tp_res_props[elem][2])
            form = et.SubElement(prop, "Format")
            scal = et.SubElement(form, "Scale",
                                 Min=tmpa0_res_to_resistor(details.attrib["Min"]),
                                 Max=tmpa0_res_to_resistor(details.attrib["Max"]),
                                 Default=tmpa0_res_to_resistor(details.attrib["Default"]),
                                 Type="xsd:unsignedInt")

            if "PermW" in details.attrib:
                prop.attrib["PermW"] = details.attrib["PermW"]

        # tpa-0 "level" property
        details = sub.find(".//Property[@PgmName='ThermProtA-0:DeratingLevel']/Format/Scale")
        prop = et.SubElement(new, "Property", DName="Derating Level", Name="tp.dl", Map="", Api="1", Unit="%")
        form = et.SubElement(prop, "Format")
        scal = et.SubElement(form, "Scale",
                             Min=tpma0_lev_to_level(details.attrib["Min"]),
                             Max=tpma0_lev_to_level(details.attrib["Max"]),
                             Default=tpma0_lev_to_level(details.attrib["Default"]),
                             Type="xsd:unsignedInt")

        if "PermW" in details.attrib:
            prop.attrib["PermW"] = details.attrib["PermW"]

        # handle/add the exceptions:
        # mode is constant = resistor-based
        prop = et.SubElement(new, "Property", DName="Mode", Name="tp.mo", Map="", Api="1", Unit="")
        form = et.SubElement(prop, "Format")
        et.SubElement(form, "Scale",
                      Default="0",
                      Min="0",
                      Max="0",
                      Type="xsd:boolean")

        # tp.or (ShutOff Resistance) is always off, i.e. not changeable
        prop = et.SubElement(new, "Property", DName="ShutOff Resistance", Name="tp.or", Map="", Api="1", Unit="Ohm")
        form = et.SubElement(prop, "Format")
        et.SubElement(form, "Scale",
                      Default="65535",
                      Min="65535",
                      Max="65535",
                      Off="65535",
                      Initial="65535",
                      Type="xsd:unsignedInt")

        # no indirect properties and no switches in ThermProtA-1

        # finally handle HwResources
        hw_resouces = sub.findall("./HwResource")
        for hwr in hw_resouces:
            new.append(deepcopy(hwr))
        return new

    sub = getsubdevice(structure, "ThermProtA-1")      # Thermal Protection based on ThermProtA-1 in NAFTA DX drivers, e.g. in 78033
    # this subdev uses artificial values that have to be transThermalformed before use
    # on UI/API, thus Map attribute is empty
    if sub is not None:

        def tpma1_lev_to_level(value):
            value = int(value)
            return str(round(value / 1023 * 100))

        def tmpa1_res_to_resistor(value):
            value = int(value)
            return str(round(value * 10000 / (65535 - value)))

        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)
        new.attrib["Mapping"] = "20"

        details = sub.find(".//Property[@PgmName='ThermProtA-1:EnableNtc']/Format/Scale")
        prop = et.SubElement(new, "Property", DName="Enable", Name="tp.en", Map="ThermProtA-1:EnableNtc", Api="1", Unit="")
        form = et.SubElement(prop, "Format")
        et.SubElement(form, "Scale",
                      Default=details.attrib["Default"],
                      Min=details.attrib["Min"],
                      Max=details.attrib["Max"],
                      Type="xsd:boolean")

        tp_res_props = {
            "ThermProtA-1:StartDeratingResistance": ("tp.sr", "Start Derating Resistance", "Ohm"),
            "ThermProtA-1:EndDeratingResistance": ("tp.er", "End Derating Resistance", "Ohm"),
        }

        # handle all tpa-1 "resistor" properties
        for elem in tp_res_props:
            details = sub.find(".//Property[@PgmName='%s']/Format/Scale" % elem)

            prop = et.SubElement(new, "Property", DName=tp_res_props[elem][1], Name=tp_res_props[elem][0],  
                                 Map="", Api="1", Unit=tp_res_props[elem][2])
            form = et.SubElement(prop, "Format")
            scal = et.SubElement(form, "Scale",
                                 Min=tmpa1_res_to_resistor(details.attrib["Min"]),
                                 Max=tmpa1_res_to_resistor(details.attrib["Max"]),
                                 Default=tmpa1_res_to_resistor(details.attrib["Default"]),
                                 Type="xsd:unsignedInt")

            if "PermW" in details.attrib:
                prop.attrib["PermW"] = details.attrib["PermW"]

        # tpa-1 "level" property
        details = sub.find(".//Property[@PgmName='ThermProtA-1:DeratingLevel']/Format/Scale")
        prop = et.SubElement(new, "Property", DName="Derating Level", Name="tp.dl", Map="", Api="1", Unit="%")
        form = et.SubElement(prop, "Format")
        et.SubElement(form, "Scale",
                      Min=tpma1_lev_to_level(details.attrib["Min"]),
                      Max=tpma1_lev_to_level(details.attrib["Max"]),
                      Default=tpma1_lev_to_level(details.attrib["Default"]),
                      Type="xsd:unsignedInt")

        if "PermW" in details.attrib:
            prop.attrib["PermW"] = details.attrib["PermW"]

        # handle/add the exceptions:
        # mode is constant = Resistor-based
        prop = et.SubElement(new, "Property", DName="Mode", Name="tp.mo", Map="", Api="1", Unit="")
        form = et.SubElement(prop, "Format")
        et.SubElement(form, "Scale",
                      Default="0",
                      Min="0",
                      Max="0",
                      Type="xsd:boolean")

        # tp.or ("ShutOff Resistance") is always off, not changeable
        prop = et.SubElement(new, "Property", DName="ShutOff Resistance", Name="tp.or", Map="", Api="1", Unit="Ohm")
        form = et.SubElement(prop, "Format")
        et.SubElement(form, "Scale",
                      Default="65535",
                      Min="65535",
                      Max="65535",
                      Off="65535",
                      Type="xsd:unsignedInt")

        # no indirect properties and no switches in ThermProtA-1
        # finally handle HwResources
        hw_resouces = sub.findall("./HwResource")
        for hwr in hw_resouces:
            new.append(deepcopy(hwr))
        return new
    sub = getsubdevice(structure, "ThermProt-YG")      # Thermal Protection based on ThermProtA-1 in NAFTA DX drivers, e.g. in 78033
    # this subdev uses artificial values that have to be transThermalformed before use
    # on UI/API, thus Map attribute is empty
    if sub is not None:

        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)
        new.attrib["Mapping"] = "0"

        prop = createFeaProperty("tp.en", "Enable", "", "", "Scale", 1, Default="1", Min="1", Max="1")
        new.append(prop)
        ddfile.createFeaPropertyFromDDProperty(new, sub, "ThermProt-YG:StartDeratingResistance", "tp.sr")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "ThermProt-YG:EndDeratingResistance", "tp.er")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "ThermProt-YG:FinalPowerReductionLevel", "tp.dl")
        # mode is constant = Resistor-based
        prop = createFeaProperty("tp.mo", "Mode", "", "", "Scale", 1, Default="0", Min="0", Max="0")
        new.append(prop)
        # tp.or ("ShutOff Resistance") is always off, not changeable
        prop = createFeaProperty("tp.or", "ShutOff Resistance", "", "Ohm", "Scale", 1, Default="65535", Min="65535", Max="65535", Off="65535")
        new.append(prop)

        return new

    if debug > 1:
        print("Info: no thermprot feature in the driver.")
    return None


def getsettings_tuning(structure, switches=None, constants=None, filename=""):
    new = et.Element("Feature", Name="tf", DName="Tuning Factor", Instances="1")
    dep = detectDependencies(structure, "tf", False, key="PwmConfig-0", cur="OTConfig-1|PwmConfig-0")
    if dep != "tf":
        new.attrib["Dep"] = dep

    tf = structure.find(".//Property[@PgmName='OTConfig-1:TuningFactor']")  # device has TF feature based on the OTConfig-1, e.g. in AM17671
    if tf is not None:
        sub = structure.find(".//SubDevice[@PgmName='OTConfig-1']")             # get source for programming attibute of TF feature
        tflimits = tf.find("./Format/Scale")
        if "Min" in tflimits.attrib:
            tfmin = int(tflimits.attrib["Min"])
        else:
            tfmin = 50

        if "Max" in tflimits.attrib:
            tfmax = int(tflimits.attrib["Max"])
        else:
            tfmax = 100

        if "Default" in tflimits.attrib:
            tfdef = int(tflimits.attrib["Default"])
        else:
            tfdef = 100

        if tfmin != tfmax:  # --> tuning factor is settable
            new.attrib["Mapping"] = "0"
            new.attrib["Programming"] = sub.attrib["Programming"]      # copy "programming" attribute from souce, see open issues %%%%%%%%%%
            copy_LockProgramming(sub, new)

            tmp = et.SubElement(new, "Property", DName="Enable", Name="tf.en", Map="", Api="1", Unit="")
            form = et.SubElement(tmp, "Format")
            if tfdef == 100:
                et.SubElement(form, "Scale", Default="0", Min="0", Max="1", Type="xsd:boolean")   # tf is disabled
            else:
                et.SubElement(form, "Scale", Default="1", Min="0", Max="1", Type="xsd:boolean")   # any other value means "enabled"

            tmp = et.SubElement(new, "Property", DName="Tuning Level (Light Output)", Name="tf.set", 
                                Map="OTConfig-1:TuningFactor", Api="1", Unit="%")
            if "PermW" in tf.attrib:
                tmp.attrib["PermW"] = tf.attrib["PermW"]
            form = et.SubElement(tmp, "Format")
            et.SubElement(form, "Scale", Default=str(tfdef), Min=str(tfmin), Max=str(tfmax), Type="xsd:unsignedByte")

            tmp = et.SubElement(new, "Property", DName="Min Limit", Name="tf.min", Map="", Api="1", Unit="%")
            if "PermW" in tf.attrib:
                tmp.attrib["PermW"] = tf.attrib["PermW"]
            form = et.SubElement(tmp, "Format")
            et.SubElement(form, "Scale", Default=str(tfmin), Min=str(tfmin), Max=str(tfmin), Type="xsd:unsignedByte")

            tmp = et.SubElement(new, "Property", DName="Max Limit", Name="tf.max", Map="", Api="1", Unit="%")
            if "PermW" in tf.attrib:
                tmp.attrib["PermW"] = tf.attrib["PermW"]
            form = et.SubElement(tmp, "Format")
            et.SubElement(form, "Scale", Default=str(tfmax), Min=str(tfmax), Max=str(tfmax), Type="xsd:unsignedByte")

            return new  # valid exit for OTConfig-1 subdevice present and TF available


    sub = structure.find(".//SubDevice[@PgmName='PwmConfig-0']")        # device has TF feature based on the PwmConfig-0, e.g. in AM10971
    if sub is not None:
        factor = structure.find(".//Property[@PgmName='PwmConfig-0:TuningFactor']")   
        tfdef = factor.find("./Format/Scale").attrib["Default"]     
        tfdef_min = factor.find("./Format/Scale").attrib["Min"]
        tfdef_max = factor.find("./Format/Scale").attrib["Max"] 

        if tfdef_min == tfdef_max:  # TF not settable -> driver has no TF
            return None 

        tfmin = structure.find(".//Property[@PgmName='PwmConfig-0:MinimumTuningFactor']")            
        tfmax = structure.find(".//Property[@PgmName='PwmConfig-0:MaximumTuningFactor']")   

        new.attrib["Mapping"] = "0"
        new.attrib["Programming"] = sub.attrib["Programming"]   
        copy_LockProgramming(sub, new)  

        tmp = et.SubElement(new, "Property", DName="Enable", Name="tf.en", Map="", Api="1", Unit="")
        form = et.SubElement(tmp, "Format")
        if tfdef == "100":
            et.SubElement(form, "Scale", Default="0", Min="0", Max="1", Type="xsd:boolean")   # tf is disabled
        else:
            et.SubElement(form, "Scale", Default="1", Min="0", Max="1", Type="xsd:boolean")   # any other value means "enabled"


        tmp = et.SubElement(new, "Property", DName="Tuning Level (Light Output)", Name="tf.set", 
                            Map="PwmConfig-0:TuningFactor", Api="1", Unit="%")
        if "PermW" in factor.attrib:
            tmp.attrib["PermW"] = factor.attrib["PermW"]
        tmp.append(copy_format_element(factor.find("./Format"), 1))


        tmp = et.SubElement(new, "Property", DName="Min Limit", Name="tf.min", 
                            Map="PwmConfig-0:MinimumTuningFactor", Api="1", Unit="%")
        if "PermW" in tfmin.attrib:
            tmp.attrib["PermW"] = tfmin.attrib["PermW"]
        tmp.append(copy_format_element(tfmin.find("./Format"), 1))


        tmp = et.SubElement(new, "Property", DName="Max Limit", Name="tf.max", 
                            Map="PwmConfig-0:MaximumTuningFactor", Api="1", Unit="%")
        if "PermW" in tfmax.attrib:
            tmp.attrib["PermW"] = tfmax.attrib["PermW"]
        tmp.append(copy_format_element(tfmax.find("./Format"), 1))

        return new  


    sub = getsubdevice(structure, "TF-0")   # TF feature based on the TF-0 subdevice
    if sub is not None:
        instances = getFeatureInstanceCount(sub, new, constants)

        new.attrib["Mapping"] = "0"
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)

        tf = sub.find("./Property[@PgmName='TF-0:Enable']")
        if tf is not None:
            tmp = et.SubElement(new, "Property", DName="Enable", Name="tf.en", Map="TF-0:Enable", Api="1", Unit="")
            tmp.append(copy_format_element(tf.find("./Format"), instances))
            tf_enabledFormat = tf.find("./Format")
        else:
            print("error, missing TF-0:Enable")
            return

        tf = sub.find("./Property[@PgmName='TF-0:TuningFactor']")
        if tf is not None:
            tmp = et.SubElement(new, "Property", DName="Tuning Level (Light Output)", Name="tf.set", 
                                Map="TF-0:TuningFactor", Api="1", Unit="%")
            tmp.append(copy_format_element(tf.find("./Format"), instances))   

            for i in range(instances):
                if tf_enabledFormat[i].attrib["Default"] == "0":       # import case (correction): if tf is disabled, set value to 100% 
                                                                       # independent of what is in source file to comply with BE parameter validation                            
                    tmp[0][i].attrib["Default"] = "100"
        else:
            print("error, missing TF-0:TuningFactor")
            return

        tf = sub.find("./Property[@PgmName='TF-0:MinimumTuningFactor']")
        if tf is not None:
            tmp = et.SubElement(new, "Property", DName="Min Limit", Name="tf.min", 
                                Map="TF-0:MinimumTuningFactor", Api="1", Unit="%")
            tmp.append(copy_format_element(tf.find("./Format"), instances))
        else:
            print("error, missing TF-0:MinimumTuningFactor")
            return

        tf = sub.find("./Property[@PgmName='TF-0:MaximumTuningFactor']")
        if tf is not None:
            tmp = et.SubElement(new, "Property", DName="Max Limit", Name="tf.max", 
                                Map="TF-0:MaximumTuningFactor", Api="1", Unit="%")
            tmp.append(copy_format_element(tf.find("./Format"), instances))
        else:
            print("error, missing TF-0:MaximumTuningFactor")
            return

        tf = sub.find("./Property[@PgmName='TF-0:ReferenceLumenOutput']")
        if tf is not None:
            tmp = et.SubElement(new, "Property", DName="Reference Luminous Flux", Name="tf.ref", 
                                Map="TF-0:ReferenceLumenOutput", Api="1", Unit="lm")
            tmp.append(copy_format_element(tf.find("./Format"), instances))

        return new   # valid exit for TF-0 subdevice

    if debug > 1:
        print("Info: No TF feature in the driver.")
    return None

def getsettings_serial(structure, switches=None, constants=None, filename=""):
    scale = structure.find(".//Property[@PgmName='Bank0:SerialNumber']/Format/Scale")
    if scale is None:
        scale = structure.find(".//Property[@PgmName='ModelInfo-0:SerialNumber']/Format/Scale")
    if scale is not None:
        serial = scale.attrib["Default"]
        tmp = et.Element("Constant", Name="SerialNumber", Value=serial)
        constants.append(tmp)

def getsettings_ctrl(structure, switches=None, constants=None, filename=""):
    sub = structure.find(".//SubDevice[@PgmName='CVP-1']")
    if sub is not None:
        new = et.Element("Feature", Name="ctrl", DName="Channel error control", Mapping="0", Instances="1")
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)
        fmt = sub.find(".//Property[@PgmName='CVP-1:ChannelLampFailureMask']/Format")
        dname = ddfile.getDisplayName(fmt.getparent())
        feaProp = et.Element("Property", Name="ctrl.chn", DName=dname, Map="CVP-1:ChannelLampFailureMask", Api="1", Unit="")
        feaProp.append(deepcopy(fmt))
        new.append(feaProp)
        return new
    
def getsettings_ygdim(structure, switches=None, constants=None, filename=""):
    sub = structure.find(".//SubDevice[@PgmName='YGDim']")
    if sub is not None:
        new = et.Element("Feature", Name="ygdim", DName="Dimming", Mapping="0", Instances="1")
        new.attrib["Programming"] = sub.attrib["Programming"]       # copy "programming" attribute from source
        copy_LockProgramming(sub, new)
        ddfile.createFeaPropertyFromDDProperty(new, sub, "YGDim:EnableAnalogDimming", "ygdim.endim")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "YGDim:DimmingCurve", "ygdim.dc")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "YGDim:DimmingMin", "ygdim.dmin")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "YGDim:LongSoftStart", "ygdim.lss")
        ddfile.createFeaPropertyFromDDProperty(new, sub, "YGDim:DimToZero", "ygdim.d2z")
        return new
        

# endregion
# region ############ "public" interface functions ############################

def convert_bytes_to_xmltree(sourcedata):

    parser = et.XMLParser(remove_blank_text=True)
    # sourcedata = bytes(sourcedata, encoding='utf-8')  # input is already bytes
    source_element = et.fromstring(sourcedata, parser)
    source_tree = et.ElementTree(source_element)

    return source_tree


def set_tree_to_default(structure):

    # fix backward compatibility issues here, so that no separte or modified convert function
    # is required for those cases

    # if structure.getroot().tag == "ECGDescription":    # the dd file case
    #     md_elem = structure.find(".//MetaData")
    # else:
    #     md_elem = structure.find(".//ECGDescription/MetaData")  # all other cases

    # scope = md_elem.find("Scope")
    # if scope is None:                   # fix the case of old PF without <scope>, to make sure that conversion is not stopped
    #                                     # BE will check if driver is supported at all...
    #     if debug > 2:
    #         print("warning: got an old file with no scope object, adding a default")
    #     tmp = et.Element("Scope")
    #     tmp.text = "C"
    #     md_elem.append(tmp)

    for elem in structure.iter():
        if "SetValue" in elem.attrib:
            elem.attrib["Default"] = elem.attrib["SetValue"]
            del(elem.attrib["SetValue"])
        if "Exponent" in elem.attrib:       # some cleanup...
            del(elem.attrib["Exponent"])
    return structure


def convert_forfamilyprogramming(sourcedata, filename=""):

    sourcedata = getdriver(sourcedata)              # get the EcgDescription of the first driver
                                                    # result is always of type Element

    scope = sourcedata.find(".//MetaData/Scope")    # make sure that sourcedata has scope=C to permit converting
    if scope is None:                       
        tmp = et.Element("Scope")
        tmp.text = "C"
        elem = sourcedata.find(".//MetaData")   
        elem.append(tmp)

    target = convert_ecgdescription_to_fea(sourcedata, filename) 

    # if type(sourcedata) == et._ElementTree:
    #     sourcedata = sourcedata.getroot()

    # handle the master key, e.g. <OEM code="0x01010101" toBeProgrammed="false" LockPartial="false" Type="2" />    
    tmp = None
    master_key = sourcedata.find(".//MetaData/OEM")
    if master_key is not None:
        if master_key.attrib["Type"]=="1" or master_key.attrib["Type"]=="2":
            tmp = master_key.attrib["code"]
            if tmp.startswith("0x"):
                tmp = str(int(tmp, 16))              # convert to string of int
            if tmp != None:
                key = target.find(".//Features/Feature/Property[@Name='key.mp']/Format/Scale")
                key.attrib["Default"] = tmp 
            en = target.find(".//Features/Feature/Property[@Name='key.en']/Format/Scale")    # enable is always there               
            if int(tmp)==0:
                en.attrib["Default"] = "0"
            else:
                en.attrib["Default"] = "1"

            if "LockPartial" in master_key.attrib:    
                pa = target.find(".//Features/Feature/Property[@Name='key.all']/Format/Scale") 
                if pa != None:
                    partial = master_key.attrib["LockPartial"]
                    if partial == "true":
                        pa.attrib["Default"]="0"
                    else:
                        pa.attrib["Default"]="1"
        else:
            print("Got an unsupported key type!")
        

    # handle the service key, e.g. <ServiceKey code="0x02020202" Type="2" />
    tmp = None
    service_key = sourcedata.find(".//MetaData/ServiceKey")
    if service_key is not None and "Type" in service_key.attrib:
        if service_key.attrib["Type"]=="1" or service_key.attrib["Type"]=="2": 
            tmp = service_key.attrib["code"]
            if tmp.startswith("0x"):
                tmp = str(int(tmp, 16))         # convert to string of int
        if tmp != None:
            key = target.find(".//Features/Feature/Property[@Name='key.sp']/Format/Scale")
            key.attrib["Default"] = tmp       
        else:
            print("Got an unsupported key type!")

 
    ## handle the permissions? --> not needed: pu and ps are already copied in standard procedure
    # permuser = sourcedata.find(".//Property[@PgmName='MSK-0:PermUser']/Format/Scale")
    # if permuser != None:
    #     permuserTarget = target.find(".//Features/Feature/Property[@Name='key.pu']/Format/Scale")
    #     permuserTarget.attrib["Default"] = permuser.attrib["Default"]


    # permservice = sourcedata.find(".//Property[@PgmName='MSK-0:PermService']/Format/Scale")
    # if permservice != None:
    #     permservTarget = target.find(".//Features/Feature/Property[@Name='key.ps']/Format/Scale")
    #     permservTarget.attrib["Default"] = permservice.attrib["Default"]


    ## all other programming options are handled by the BE and not the converter...
    ## thus the following code is not used here anymore

    # po = et.Element("ProgrammingOptions")       # create a new section in fea for "programming options"
    # sre = sourcedata.find(".//MetaData/Rework")
    # if sre != None:
    #     re = et.SubElement(po, "Rework")
    #     re.attrib["code"] = sre.attrib["code"]
    #     re.attrib["selected"] = sre.attrib["selected"]        
    
    # svp = sourcedata.find(".//MetaData/VerifyParameters")
    # if svp != None:
    #     vp = et.SubElement(po, "VerifyParameters")
    #     vp.text = svp.text
    #
    # sie = sourcedata.find(".//MetaData/IsEnableSingleProgramming")
    # if sie != None:
    #     ie = et.SubElement(po, "IsEnableSingleProgramming")
    #     ie.text = sie.text
    #
    # sfp = sourcedata.find(".//MetaData/FamilyProgramming")
    # if sfp != None:
    #     fp = et.SubElement(po, "FamilyProgramming")  
    #     fp.text = sfp.text
    #
    # slo = sourcedata.find(".//MetaData/LockSettings")
    # if slo != None:
    #     lo = et.SubElement(po, "LockSettings")   
    #     lo.text = slo.text
    #
    # md = target.find(".//MetaData")
    # md.addnext(po)

    return target

def brand_featurenames(fea):
    # first look for branded opmodes
    isBranded = False
    modes = fea.findall(".//Mode")
    brandings = ["midnight", "bilevel", "amp dim"]
    for mode in modes:      # iterate over op modes
        if "DName" in mode.attrib:   # only handle modes with a name. Other modes are "spacer entries", hence skip them here
            om_name = mode.attrib["DName"].strip().lower()
            if any(substring in om_name for substring in brandings):
                isBranded = True
                break
    # if branding detectet, brand feature names
    if isBranded:
        features = fea.findall(".//Feature")
        for feature in features:
            dname = str(feature.attrib["DName"])
            dname = dname.replace("AstroDIM", "Midnight")
            dname = dname.replace("StepDIM", "Bilevel")
            dname = dname.replace("MainsDIM", "Amp Dim")
            feature.attrib["DName"] = dname
    return

scriptDir = os.path.dirname(__file__)
with open (os.path.join(scriptDir, "featuredescriptions.fdesc"), "rb") as reader:
    featureDescriptions = reader.read()
genericFeatureHandler = GenericFeatureHandler(featureDescriptions)

def convert_ecgdescription_to_fea(sourcedata, filename=""):

    resetGlobalVariables()
    source = getdriver(sourcedata)      # get the first driver in case of PF with multiple drivers included
                                        # returned data type is "lxml.etree._Element"

    if debug > 1:
        print("Saving the updated inputfile")
        sourcedata.write(open("~tmp.xml", "bw"), pretty_print=True, xml_declaration=True, encoding='utf-8')

    if type(source) == et._ElementTree:
        source = source.getroot()

    version = int(source.attrib["Version"])
    if version<4:
        raise NameError ("Version is too old.")
    if version>5:
        raise NameError ("Version is newer than expected.")

    tmp = source.find(".//MetaData/Scope")      # make sure that only files with scope==c are converted.  
                                                # important is to exclude old dd files during batch creation of feas
    if tmp is not None:             # creation of fea: source dd files have scope element -> keep as is 
        scope = tmp.text            # and stop if C is not included
    else:                       
        scope = "C"                 # import case: some old files may have no scope, thus add "C" to continue

    if "C" not in scope and "F" not in scope:
        if debug > 1:
            print("Info: skipping driver %s which is not scope of T4T-Cloud. Scope of current driver is " % fn, scope)
        return

    target = et.Element("FeatureFile")
    root = et.ElementTree(target)   # start a new data structure for the target driver

    md = et.Element("MetaData")
    target.append(md)

    om = et.Element("OperatingModes")
    target.append(om)

    sw = et.Element("Switches")     # Switches are directly controlled by the Operating Mode,
    # but are not available in the API as switches are set by OpsMode control
    # No dedicated validation is required for switches, as ops modes are correct by definition! (ensured during dd creation process)
    target.append(sw)

    co = et.Element("Constants")
    target.append(co)

    fill_metadata(source, metadata=md, constants=co)

    perm = source.find(".//Permissions")
    if perm is not None:
        target.append(deepcopy(perm))
    else:
        perm = et.Element("Permissions")   # in case of no perms in the dd file, include an empty Permissions element
        target.append(perm)

    fea = et.Element("Features")
    target.append(fea)
    featurelist = []

    feature_getters = [         # list of all functions to extract the features from a dd file
        getsettings_dali,       # all dali must be first
        getsettings_dali_dt8,
        getsettings_dali_addressing,
        getsettings_current,
        getsettings_clm,
        getsettings_d2w,
        getsettings_tuning,
        getsettings_astro,
        getsettings_astrotw,
        getsettings_sd,
        getsettings_sdtw,
        getsettings_pd,
        getsettings_pdtw,
        getsettings_md,
        getsettings_0t010,
        getsettings_sso,
        getsettings_d2d,
        getsettings_dimmingmode,
        getsettings_emergency,
        getsettings_dexal,
        getsettings_eol,
        getsettings_opstime,
        getsettings_thermprot,
        getsettings_driverguard,
        getsettings_permShutdown,
        getsettings_tdcf,
        getsettings_luminfo,
        getsettings_configlock,
        getsettings_ledmodule,
        getsettings_serial,
        getsettings_rgbw,
        getsettings_ctrl,
        getsettings_ygdim,
        getsettings_pod,
        getsettings_spd,
        getsettings_rpt,
        getsettings_dpwm
    ]

    subdevice2featureMap = {}
    for func in feature_getters:
        feature = func(source, switches=sw, constants=co, filename=filename)
        if feature is not None:
            fea.append(feature)
            featurelist.append(feature.attrib["Name"])
    # now handle generic features definde by a xml feature description, either global from a xml file in the converter
    # or optionally from feature description in the ddfile itself
    genericFeatures = genericFeatureHandler.generateFeaturesFromDDFile(source, switches=sw, constants=co, filename=filename)
    for feature in genericFeatures:
            fea.append(feature)
            featurelist.append(feature.attrib["Name"])
    
    lockedFeatures = getPartialOemKeyLockedFeatures(source)
    if lockedFeatures is not None:
        fea.attrib["PartialLocked"] = lockedFeatures

    fill_operatingmodes(genericFeatureHandler, source, om=om, features=fea, feanames=featurelist, filename=filename)

    # ### save featurefile for checking 
    # fn = fn[:-3]+"fea"  #  remove xml and add fea as extension
    # root.write(open(os.path.join(os.path.join(path,"../feas"), fn),"bw"), pretty_print=True, xml_declaration=True, encoding='utf-8')

    # temporary hack to brand feature names without dd file support
    brand_featurenames(root)
    
    return root


# endregion
# region ############ "tests" for this module #################################

if __name__ == "__main__":      # to prevent that this code is  executed when this file is imported 
                                # as module "converter", e.g. when used in the T4T-Cloud BE

    debug = 0
    default_path = r'C:\Work\T4TWeb\artifacts\ECGs'
    default_path = os.getenv('artifacts', default_path)
    if len(sys.argv) > 1:
        mypath = os.path.abspath(sys.argv[1])
    else:
        mypath = input("Where are the dd files [%s]? "%default_path)
        if mypath == "":
            mypath = default_path

    if os.path.isfile(mypath):
        fn = mypath
        with open (fn, "rb") as fh:
            rawdata = fh.read()

        xmldata = convert_bytes_to_xmltree(rawdata)
        xmldata = set_tree_to_default(xmldata)
        targetdata = convert_ecgdescription_to_fea(xmldata, filename=fn)

        if targetdata != None:
            fn = fn[:-3]+"fea"  #  remove xml and add fea as extension
            targetdata.write(open(fn,"bw"), pretty_print=True, xml_declaration=True, encoding='utf-8')
        else:
            print("error: did not get a feature file back")
        sys.exit(0)

    if not os.path.isdir(mypath): 
        print("No valid directory provided!")
        sys.exit(0)

    start = time.time()     # and go...
    _, _, filenames = next(os.walk(mypath), (None, None, []))
    for fn in filenames:

        if fn.endswith(".xml"):
            if debug > 1:
                print("*** processing: ", fn)

            # if fn == "an00954.xml":   # for setting breakpoints
            #     print("*** processing: ", fn)

            with open(os.path.join(mypath, fn), "rb") as fh:
                rawdata = fh.read()

            xmldata = convert_bytes_to_xmltree(rawdata)

            targetdata = convert_ecgdescription_to_fea(xmldata, filename=fn)
            #targetdata = convert_forfamilyprogramming(xmldata, filename=fn)
            if targetdata is not None:

                fn = fn[:-3] + "fea"  # remove xml and add fea as extension
                targetdata.write(open(os.path.join(os.path.join(mypath, "../feas"), fn), "bw"),
                                 pretty_print=True, xml_declaration=True, encoding='utf-8')

    end = time.time()
    print("Done in %.3f s" % (end - start))

# endregion
