from lxml import etree as et
from typing import List, Set, Dict, Tuple
import os
import sys
import time
import ddfile

class Feature:
    def __init__(self, name : str, dname : str, dep : str, map : str =0, inst : str ="1") -> None:
        self.name = name            # the features name
        self.dname = dname          # display name
        self.mapping = map          # mapping id
        self.instances = inst       # how many instances
        self.subdevice = ""         # the 'main' subdevice for the feature
        self.props2map = []         # the property mappings, e.g. subdevice prop name to fea prop name as a list with two elements
        self.values2map = []        # 
        self.dependencies = dep     # a comma delimited list of fea names as dependencies
        self.switches = []          # a list of all switches, parsed from a comma seperated string in the xml
        self.opmodes = ""           # a comma delimited list of opmodes where the feature is active
        self.omRefProp = ""         # indicates a property with a given value where the feature is active
        self.omRefValue = ""        # indicates a property with a given value where the feature is active
        self.omDeps = {}            # opmode feature dependencies

    def addPropertyMappings(self, propMappings : str):
        """Adds property mappings to a feature object.
        
        Format: subdevice_property_name|fea_property_name;subdevice_property_name|fea_property_name;..."""

        propMappingList = propMappings.split(';')
        for pm in propMappingList:
            mapping = pm.split('|')
            self.props2map.append(mapping)
        self.subdevice = self.props2map[0][0].split(':')[0]

    def detectFeature(self, ddStructure, switches, constants, filename) -> bool:
        """Detects if a feature is present in the current ddfile."""

        # just use the first property mapping for detecting the presence
        propName = self.props2map[0][0]
        ddProp = ddStructure.find(f".//Property[@PgmName='{propName}']")
        return ddProp is not None

    def createFeature(self, ddStructure, switches, constants, filename):
        """Creates a fea feature xml tree for the given ddfile xml tree and returns a fea feature tree"""

        new = et.Element("Feature", Name=self.name, DName=self.dname, Mapping=self.mapping, Instances=self.instances)
        if self.dependencies != "":
            new.attrib["Dep"] = self.dependencies
        sub = ddStructure.find(".//SubDevice[@PgmName='%s']" % self.subdevice)
        new.attrib["Programming"] = sub.attrib["Programming"]
        # todo config lock mapping and subdevice to feature mapping
        for prop in self.props2map:
            ddfile.createFeaPropertyFromDDProperty(new, sub, prop[0], prop[1])
        # simple value mappings
        for vals in self.values2map:
            feaPropFmt = self.__getFeaPropFormat(new, vals[0])
            self.__doValueMappings(feaPropFmt, vals[1], vals[2])
        # handle switches if any
        for sw in self.switches:
            val = self.__getPropertyValue(sub, sw)
            et.SubElement(switches, "Switch", Name=sw, Map=sw, SetValue=val)
        return new
    
    def isFeatureDisabled(self, featureName : str, opModeName : str, refProp : str, refValue : str) -> bool:
        if self.name != featureName:
            return False
        if self.omRefProp == refProp and self.omRefValue != refValue:
            return True
        if len(self.opmodes) > 0:
            omList = self.opmodes.split(',')
            if opModeName not in omList:
                return True
        return False
    
    def getOpModeDependencies(self) -> Dict[str, any]:
        return self.omDeps
    
    def __getPropertyValue(self, sub, propName):
        data = sub.find(f".//Property[@PgmName='{propName}']/Format")
        val = data[0].attrib["Default"]
        return val
    
    def __getFeaPropFormat(self, featureTree, propName):
        feaPropFmt = featureTree.find(f".//Property[@Name='{propName}']/Format")
        if feaPropFmt is None:
            print(f"__getFeaPropFormat: Property {propName} not found.")
        return feaPropFmt
    
    def __doValueMappings(self, feaPropFmt, srcValues, dstValue):
        srcValuesArr = srcValues.split(',')
        for valueXml in feaPropFmt:
            orgValue = valueXml.attrib["Default"].strip()
            for srcVal in srcValuesArr:
                if (orgValue == srcVal.strip()):
                    valueXml.attrib["Default"] = dstValue
                    return


class GenericFeatureHandler:
    """Handles automatic feature generation defined by a feature description xml.
    
    There are two sources of featue descriptions, one is a fixed deployed with the feature converter.
    The other one is a temporary in the ddfile."""

    def __init__(self, featureDescriptionsXml : str) -> None:
        featDescsParsed = self.__parseXml(featureDescriptionsXml)
        self._featureDescriptions = self.__loadFeatureDescriptions(featDescsParsed)

    def __loadFeatureDescriptions(self, xmlTree) -> List[Feature]:
        # load all feature descriptions from xml tree
        featDescs = xmlTree.findall(".//FeatureDescription")
        if len(featDescs) == 0:
            return []
        features = []
        for desc in featDescs:
            feature = self.__parseFeature(desc)
            features.append(feature)
        return features
    
    def __parseFeature(self, desc) -> Feature:
        name = desc.attrib["Name"].strip()
        dname = desc.attrib["DName"].strip()
        mapping = desc.attrib["Mapping"].strip()
        instances = desc.attrib["Instances"].strip()
        depElem = desc.find(".//Dependencies")
        dependencies = ""
        if depElem is not None:
            dependencies = depElem.text.strip()
        feature = Feature(name, dname, dependencies, mapping, instances)
        pms = []
        propMappings = desc.findall(".//PropertyMapping")
        for pm in propMappings:
            pms.append(pm.text.strip())
        propMappingStr = ";".join(pms)
        feature.addPropertyMappings(propMappingStr)
        vms = []
        valueMappings = desc.findall(".//ValueMapping")
        for vmXml in valueMappings:
            mapping = vmXml.text.strip().split("->")
            vm = [vmXml.attrib["Prop"].strip(), mapping[0].strip(), mapping[1].strip()]
            vms.append(vm)
        feature.values2map = vms
        active4OpMode = desc.find(".//FeatureAvailable")
        if active4OpMode is not None:
            feature.opmodes = active4OpMode.attrib["OpModes"]
            propValue = active4OpMode.attrib["OpModeRef"].split('=')
            if len(propValue) == 2:
                feature.omRefProp = propValue[0]
                feature.omRefValue = propValue[1]
        switchesXml = desc.find(".//Switches")
        if switchesXml is not None:
            feature.switches = switchesXml.text.strip().split(',')
        opModeXml = desc.find(".//OpModeDeps")
        if opModeXml is not None:
            omDepsList = opModeXml.text.strip().split(',')
            for dep in omDepsList:
                feature.omDeps[dep] = 1
        return feature

    # Returns a list of feature xmls
    def generateFeaturesFromDDFile(self, ddStructure, switches=None, constants=None, filename="") -> List[any]:
        """Generates the fea features for the given ddfile xml in ddStructure.
        
        Returns a list of fea feature xml trees"""

        # try to find some ddfile local featuredescriptions
        ddfileLocalFeatureDescriptions = self.__loadFeatureDescriptions(ddStructure)
        self.driverFeatureDescriptions = self.__mergeGlobalAndLocalFeatureDescriptions(self._featureDescriptions, ddfileLocalFeatureDescriptions)

        features = self.__generateFeatures(self.driverFeatureDescriptions, ddStructure, switches, constants, filename)

        return features

    def featureNotAvailableForOpMode(self, featureName : str, opModeName : str, refProp : str, refValue : str) -> bool:
        """Returns true if the given feature is not available for the current opmode"""

        if self.__featureNotAvailable(self.driverFeatureDescriptions, featureName, opModeName, refProp, refValue):
            return True
        return False
    
    def getOpModeDependencies(self, availableFeaturesInDriver : List[str]) -> Dict[str,any]:
        """Returns OpMode feature dependencies from all generic features as a dictionary."""

        opModeDeps = self.__getOpModeDependenciesFromFeatureDescs(self.driverFeatureDescriptions)
        # now remove all entries for features which does not exist in the current driver
        for fea in list(opModeDeps.keys()):
            if fea not in availableFeaturesInDriver:
                del opModeDeps[fea]
        return opModeDeps
    
    def __mergeGlobalAndLocalFeatureDescriptions(self, globalFeatureDescriptions : List[Feature], localFeatureDescriptions : List[Feature]) -> List[Feature]:
        """Merges the global and ddfile local feature descriptions, but allows overwriting global ones with a local one"""
        mergedList = []
        localFeatDescNames = set(map(lambda f: f.name, localFeatureDescriptions))
        # now insert global descriptions if they are not in the local one
        for fd in globalFeatureDescriptions:
            if fd.name not in localFeatDescNames:
                mergedList.append(fd)
        mergedList += localFeatureDescriptions
        return mergedList
    
    def __getOpModeDependenciesFromFeatureDescs(self, featuresDescriptions) -> Dict[str,any]:
        opModeDeps = {}
        for fDesc in featuresDescriptions:
            opModeDeps.update(fDesc.getOpModeDependencies())
        return opModeDeps
    
    def __generateFeatures(self, featuresDescriptions, ddStructure, switches, constants, filename):
        """Generates features for a given list of generic features, returns a list of feature xml trees"""

        features = []
        existingFeatures = []
        for fDesc in featuresDescriptions:
            if fDesc.detectFeature(ddStructure, switches, constants, filename):
                featXml = fDesc.createFeature(ddStructure, switches, constants, filename)
                features.append(featXml)
                existingFeatures.append(fDesc.name)
        self.__removeDependenciesIfFeaturesDoNotExist(features, existingFeatures)
        return features

    def __removeDependenciesIfFeaturesDoNotExist(self, features, existingFeatures : List[str]):
        """Removes dependencies from features if the feature does not exist in the current driver."""
        for feature in features:
            if "Dep" in feature.attrib:
                deps = feature.attrib["Dep"].split(',')
                newDeps = []
                for dep in deps:
                    if dep in existingFeatures:
                        newDeps.append(dep)
                if len(newDeps) > 0:
                    feature.attrib["Dep"] = ','.join(newDeps)
                else:
                    del feature.attrib["Dep"]
    
    def __featureNotAvailable(self, featuresDescriptions, featureName : str, opModeName : str, refProp : str, refValue : str) -> bool:
        for fDesc in featuresDescriptions:
            if fDesc.isFeatureDisabled(featureName, opModeName, refProp, refValue):
                return True
        return False
    
    def __parseXml(self, xml : str):
        """Parses the given xml string and returns a xml tree"""

        parser = et.XMLParser(remove_blank_text=True)
        source_element = et.fromstring(xml, parser)
        source_tree = et.ElementTree(source_element)

        return source_tree

