
###############################################################################
##  1.12.3
##  Task:       get the driver content in raw format and convert it to PF and 
##              then to fea file (to be used in T4T-BE)
##    
##  to do:      * test if check for writable / read-only properties is ok...
##          
##  dont do:    * no check if parameter is out of range after copy: this is done by the BE,
##                i.e. don't re-implementation validation!
##
##  Status:     released
##
###############################################################################

# region ###################  common stuff ####################################
toolname = "raw to production file converter"
toolversion = "1.0"

import binascii
import struct
import logging
import os
import sys
import math

import certifi
import ssl
import urllib3
from lxml import etree as et

#sys.path.append('../')  # make sure that lib in other dir is accessible
#import osrtup
import err

#logging.basicConfig(filename='covertfromraw.log', filemode='a', format='%(asctime)s:%(name)s:%(levelname)s:%(message)s')
## no logging to file
logger = logging.getLogger('raw2fea')  
logger.setLevel(logging.WARNING)

fh = logging.StreamHandler(sys.stdout)
fh.setLevel(logging.WARNING)
formatter = logging.Formatter('%(asctime)s:%(name)s:%(levelname)s:%(message)s')
fh.setFormatter(formatter)
logging.getLogger('').addHandler(fh)    # attach this log-hander (=logging output) to the "root logger", 
                                        # then it gets all messages from this module 

# sets the property's value
# formatElement is one of the supported format elements (scale, enum, ...)
def setPropertyValue(formatElement, value):
    formatElement.attrib["Default"] = str(value)

# endregion
# region ###################  interface specific handlers  ####################
class RawHandler():    
    def __init__(self, data):
        self.data = data
        self.interfacetype = ""  # to be set by derived class
        self.instances = 4      # quick and dirty solution  
        return

    def getInstances(self) -> int:
        return self.instances

    def getddfile(self, baseurl, apikey, gtin=0, fw=0, hw=0, model=0):
        if baseurl.lower().startswith("http://"):
            http = urllib3.PoolManager()
        else:
            http = urllib3.PoolManager(cert_reqs=ssl.CERT_REQUIRED, ca_certs=certifi.where())  
        url = baseurl+'/ddstore/api/v1/ddfile?gtin=%s&fwVersion=%s&hwVersion=%s&format=dd'%(str(gtin), str(fw), str(hw))
        if model!=0:
            url = baseurl+'/ddstore/api/v1/ddfile/modelid/%s?format=dd'%model
        headers  = {"Accept": "application/json", "X-API-KEY": apikey}
        print("DDStore call: " + url)
        res = http.request('GET', url, headers=headers)

        if res.status != 200:
            print(res.status)
            print(res.data)
            logger.Debug("No matching dd file available")   # caution: 404 from ddstore does not seem to endup here!!!
            return None 

        parser = et.XMLParser(remove_blank_text=True)
        ddfile = et.fromstring(res.data, parser)
        destfile = et.ElementTree(ddfile)
        return destfile

    def setValuefromRaw(self, raw, prop, instance=0) -> None:
        # raw is the string with the raw data
        # if <= 8 byte: raw is str of signed int
        # else raw is hexbytes (without leading 0x)
        # prop is the pointer to the property element within target structure

        pname = prop.attrib["PgmName"]
        format = prop.find(".//Format")
        for sub in format:  

            use_this_format_element = False
            if "Instance" not in sub.attrib:
                if instance==0:
                    use_this_format_element = True 
            elif int(sub.attrib["Instance"])==instance:
                use_this_format_element = True

            if use_this_format_element==True:
                # get the datatype for this property from dd file
                datatype = None
                #print(prop.attrib["PgmName"])

                if sub.tag=="Scale" or sub.tag=="EnumValues" or sub.tag=="ExpValues":
                    if "Type" in sub.attrib:
                        datatype = sub.attrib["Type"]
                if sub.tag == "HexBytes":
                    datatype = "hex"  # hexbyte has no explicit Type attribute
                if datatype == None:
                    print("error 1, no valid data format in property %s"%prop.attrib["PgmName"])

                if datatype == "xsd:floatSingle":   
                    # raw is string, e.g. "bfa66666"
                    tmp = int(raw).to_bytes(4, byteorder="big",signed=False)
                    raw = str(struct.unpack('>f', tmp)[0])
                    # struck.unpack returns returns tuple, e.g. (1.2999,), thus use result[0] to get value
                    # result[0] is float, i.e. additionally convert to str               
                setPropertyValue(sub, raw)
        return
                # online converting: https://www.h-schmidt.net/FloatConverter/IEEE754.html

    # allows calculation of property values depending on other properties
    def mapProperties(self, properties) -> None:
        return

class OsrHandler(RawHandler):
    def __init__(self, prop):
        super().__init__(prop)
        self.interfacetype="osrser" 
        self.rawbuffer = {}

        mem = self.data.find(".//Mem")
        self.rawbuffer[0] = bytearray.fromhex(mem.attrib["Data"])
        return

    def getidentification(self):
        data = self.data.find(".//Header")
        return (data.attrib["Gtin"], '', '')

    def getddfile(self, baseurl, apikey, driverid):
        return super().getddfile(baseurl, apikey, model=driverid[0])

    def getRaw4Property(self, prop, instance=0, signed=False) -> bytes:
        
        # if prop.attrib["PgmName"]=="Bank1:ContentFormatVersion":
        #     print("start debug")
        allocs = prop.findall(".//Allocation")

        for alloc in allocs:

            if "Instance" not in alloc.attrib:   # no instance attrib means instance 0
                this_is_instance = 0
            else:
                this_is_instance = int(alloc.attrib["Instance"])

            if this_is_instance == instance:
                bank      = 0
                offset    = int(alloc.find("MemAddress").attrib["Byte"])
                bytelen   = int(alloc.find("Size").attrib["ByteCount"])
                try:
                    bitoffset = int(alloc.find("MemAddress").attrib["Bit"])
                except:
                    bitoffset = 0
                bitlen    = int(alloc.find("Size").attrib["BitCount"])

                if "NetworkByteOrder" in alloc.attrib:
                    tmp = to_bool(alloc.attrib["NetworkByteOrder"]) 
                    byteorder = "big" if tmp==True else "little"
                else: 
                    byteorder = "big"   # default case, when no byteorder is given in the source PF file  
                
                if bitlen == 0:
                    bytemode = True
                else:
                    bytemode = False

                    if bitlen >8:
                        print("error: bitwise property handling larger than one byte is not supported in this version.")
                        sys.exit()  

                    if bytelen>0:
                        print("error in datamodel: mixed bit and byte mode not supported")
                        sys.exit() 

                if bytemode:
                    rawdata = self.rawbuffer[bank][offset:offset+bytelen] 
                    if bytelen<=8:
                        rawstr = str(int.from_bytes(rawdata, byteorder, signed=signed))
                    else:
                        rawstr = rawdata.hex()
                else:  # bitmode
                    rawint = self.rawbuffer[bank][offset] >> bitoffset       # implicit conversion to int 
                    mask = (2 ** bitlen) - 1
                    rawint = rawint & mask
                    if signed==True: 
                        rawint = subbyte_to_int(rawint, bitlen)
                    rawstr = str(rawint)
                return rawstr
        return None

    def copyDaliProperties(self, target, instance):
        return 

class Nfc1Handler(RawHandler):
    def __init__(self, data):
        super().__init__(data)
        self.interfacetype="nfc1" 
        self.rawbuffer = {}

        lines = self.data.findall(".//Toc")
        for line in lines:
            index = int(line.attrib["Index"])
            self.rawbuffer[index] = bytearray.fromhex(line.attrib["Data"])
        return


    def getidentification(self):
        data = self.data.find(".//Header")
        return (data.attrib["Gtin"], data.attrib["Fw"], data.attrib["Hw"])


    def getddfile(self, baseurl, apikey, driverid):
        return super().getddfile(baseurl, apikey, gtin=driverid[0], fw=driverid[1], hw=driverid[2])


    def getRaw4Property(self, prop, instance=0, signed=False) -> bytes:
        
        # if prop.attrib["PgmName"]=="Bank1:ContentFormatVersion":
        #     print("start debug")
        allocs = prop.findall(".//Allocation[@Interface]")

        for alloc in allocs:

            if "Instance" not in alloc.attrib:   # no instance attrib means instance 0
                this_is_instance = 0
            else:
                this_is_instance = int(alloc.attrib["Instance"])

            if this_is_instance == instance and  alloc.attrib["Interface"]=="NFC":
                bank      = 0
                offset    = int(alloc.find("MemAddress").attrib["Byte"])
                bytelen   = int(alloc.find("Size").attrib["ByteCount"])
                try:
                    bitoffset = int(alloc.find("MemAddress").attrib["Bit"])
                except:
                    bitoffset = 0
                bitlen    = int(alloc.find("Size").attrib["BitCount"])

                byteorder = "big"
                
                if bitlen == 0:
                    bytemode = True
                else:
                    bytemode = False

                    if bitlen >8:
                        print("error: bitwise property handling larger than one byte is not supported in this version.")
                        sys.exit()  

                    if bytelen>0:
                        print("error in datamodel: mixed bit and byte mode not supported")
                        sys.exit() 

                if bytemode:
                    rawdata = self.rawbuffer[bank][offset:offset+bytelen] 
                    if bytelen<=8:
                        rawstr = str(int.from_bytes(rawdata, byteorder, signed=signed))
                    else:
                        rawstr = rawdata.hex()
                else:  # bitmode
                    rawint = self.rawbuffer[bank][offset] >> bitoffset       # implicit conversion to int 
                    mask = (2 ** bitlen) - 1
                    rawint = rawint & mask
                    if signed==True: 
                        rawint = subbyte_to_int(rawint, bitlen)
                    rawstr = str(rawint)
                return rawstr

        return None


    def copyDaliProperties(self, target, instance):
        daliprops = target.findall(".//DALIProperties/Property") 
        for daliprop in daliprops:
            propvalue = self.getRaw4Property(daliprop, instance)  
            if propvalue != None:

                format = daliprop.find(".//Format")
                for sub in format:  
                    use_this_format_element = False
                    if "Instance" not in sub.attrib:
                        if instance==0:
                            use_this_format_element = True 
                    elif int(sub.attrib["Instance"])==instance:
                        use_this_format_element = True

                    if use_this_format_element==True:       # get the datatype for this property from dd file                          
                        datatype = None
                        if sub.tag=="Scale" or sub.tag=="EnumValues" or sub.tag=="ExpValues":
                            datatype = sub.attrib["Type"]   
                        if datatype == None:
                            print("error 2, no valid data format in property %s"%daliprop.attrib["PgmName"])
                        setPropertyValue(sub, propvalue)
        return 


class Nfc2Handler(RawHandler):
    def __init__(self, data):
        super().__init__(data)
        self.interfacetype="nfc2" 
        self.rawbuffer = {}

        lines = self.data.findall(".//Toc")
        for line in lines:
            index = int(line.attrib["Index"])
            self.rawbuffer[index] = bytearray.fromhex(line.attrib["Data"])
        return


    def getidentification(self):
        data = self.data.find(".//Header")
        return (data.attrib["Gtin"], data.attrib["Fw"], data.attrib["Hw"])


    def getddfile(self, baseurl, apikey, driverid):
        return super().getddfile(baseurl, apikey, gtin=driverid[0], fw=driverid[1], hw=driverid[2])


    def getRaw4Property(self, prop, instance=0, signed=False) -> bytes:
        
        # if prop.attrib["PgmName"]=="Bank1:ContentFormatVersion":
        #     print("start debug")
        allocs = prop.findall(".//Allocation[@Interface]")

        for alloc in allocs:

            if "Instance" not in alloc.attrib:   # no instance attrib means instance 0
                this_is_instance = 0
            else:
                this_is_instance = int(alloc.attrib["Instance"])

            if this_is_instance == instance and  alloc.attrib["Interface"]=="NFC":
                bank      = int(alloc.find("MemAddress").attrib["Bank"])
                offset    = int(alloc.find("MemAddress").attrib["Byte"])
                bytelen   = int(alloc.find("Size").attrib["ByteCount"])
                try:
                    bitoffset = int(alloc.find("MemAddress").attrib["Bit"])
                except:
                    bitoffset = 0
                bitlen    = int(alloc.find("Size").attrib["BitCount"])

                if "NetworkByteOrder" in alloc.attrib:
                    tmp = to_bool(alloc.attrib["NetworkByteOrder"]) 
                    byteorder = "big" if tmp==True else "little"
                else: 
                    byteorder = "big"   # default case, when no byteorder is given in the source PF file  
                
                if bitlen == 0:
                    bytemode = True
                else:
                    bytemode = False

                    if bitlen >8:
                        print("error: bitwise property handling larger than one byte is not supported in this version.")
                        sys.exit()  

                    if bytelen>0:
                        print("error in datamodel: mixed bit and byte mode not supported")
                        sys.exit() 

                if bytemode:
                    rawdata = self.rawbuffer[bank][offset:offset+bytelen] 
                    if bytelen<=8:
                        rawstr = str(int.from_bytes(rawdata, byteorder, signed=signed))
                    else:
                        rawstr = rawdata.hex()
                else:  # bitmode
                    rawint = self.rawbuffer[bank][offset] >> bitoffset       # implicit conversion to int 
                    mask = (2 ** bitlen) - 1
                    rawint = rawint & mask
                    if signed==True: 
                        rawint = subbyte_to_int(rawint, bitlen)
                    rawstr = str(rawint)
                return rawstr

        return None


    def copyDaliProperties(self, target, instance):
        
        daliprops = target.findall(".//DALIProperties/Property") 
        for daliprop in daliprops:
            propvalue = self.getRaw4Property(daliprop, instance)  
            if propvalue != None:

                format = daliprop.find(".//Format")
                for sub in format:  
                    use_this_format_element = False
                    if "Instance" not in sub.attrib:
                        if instance==0:
                            use_this_format_element = True 
                    elif int(sub.attrib["Instance"])==instance:
                        use_this_format_element = True

                    if use_this_format_element==True:       # get the datatype for this property from dd file                          
                        datatype = None
                        if sub.tag=="Scale" or sub.tag=="EnumValues" or sub.tag=="ExpValues":
                            datatype = sub.attrib["Type"]   
                        if datatype == None:
                            print("error 2, no valid data format in property %s"%daliprop.attrib["PgmName"])
                        setPropertyValue(sub, propvalue)
        return 

class Nfc3Handler(RawHandler):
    def __init__(self, data):
        super().__init__(data)
        self.interfacetype="nfc3" 
        self.rawbuffer = {}

        lines = self.data.findall(".//Toc")
        for line in lines:
            index = int(line.attrib["Index"])
            self.rawbuffer[index] = bytearray.fromhex(line.attrib["Data"])
        return


    def getidentification(self):
        data = self.data.find(".//Header")
        if "SerialNumber" in data.attrib:
            sn = int(data.attrib["SerialNumber"], 16)
            sn = str(sn)
        else:
            sn = "0"
        return (data.attrib["Gtin"], data.attrib["Fw"], data.attrib["Hw"], sn)

    def addSerialNumber2xml(self, ddxml, sn) -> None:
        subd = ddxml.find(".//SubDevices")
        if subd is not None:
            bank0 = et.SubElement(subd, "SubDevice", PgmName="Bank0")
            prop = et.SubElement(bank0, "Property", PgmName="Bank0:SerialNumber", AccessType="R")
            fmt = et.SubElement(prop, "Format")
            et.SubElement(fmt, "Scale", Default=sn)

    def getddfile(self, baseurl, apikey, driverid):
        ddxml = super().getddfile(baseurl, apikey, gtin=driverid[0], fw=driverid[1], hw=driverid[2])
        self.addSerialNumber2xml(ddxml, driverid[3])
        return ddxml

    def getRaw4Property(self, prop, instance=0, signed=False) -> bytes:
        
        # if prop.attrib["PgmName"]=="Bank1:ContentFormatVersion":
        #     print("start debug")
        allocs = prop.findall(".//Allocation[@Interface]")

        for alloc in allocs:

            if "Instance" not in alloc.attrib:   # no instance attrib means instance 0
                this_is_instance = 0
            else:
                this_is_instance = int(alloc.attrib["Instance"])

            if this_is_instance == instance and  alloc.attrib["Interface"]=="NFC":
                bank      = int(alloc.find("MemAddress").attrib["Bank"])
                offset    = int(alloc.find("MemAddress").attrib["Byte"])
                bytelen   = int(alloc.find("Size").attrib["ByteCount"])
                try:
                    bitoffset = int(alloc.find("MemAddress").attrib["Bit"])
                except:
                    bitoffset = 0
                bitlen    = int(alloc.find("Size").attrib["BitCount"])

                # there is a bug in the driver, byte order is always little endian, independent what the xml says
                byteorder = "little"
                
                if bitlen == 0:
                    bytemode = True
                else:
                    bytemode = False

                    if bitlen >8:
                        print("error: bitwise property handling larger than one byte is not supported in this version.")
                        sys.exit()  

                    if bytelen>0:
                        print("error in datamodel: mixed bit and byte mode not supported")
                        sys.exit() 

                if bytemode:
                    rawdata = self.rawbuffer[bank][offset:offset+bytelen] 
                    if bytelen<=8:
                        rawstr = str(int.from_bytes(rawdata, byteorder, signed=signed))
                    else:
                        rawstr = rawdata.hex()
                else:  # bitmode
                    rawint = self.rawbuffer[bank][offset] >> bitoffset       # implicit conversion to int 
                    mask = (2 ** bitlen) - 1
                    rawint = rawint & mask
                    if signed==True: 
                        rawint = subbyte_to_int(rawint, bitlen)
                    rawstr = str(rawint)
                return rawstr

        return None


    def copyDaliProperties(self, target, instance):
        return 

    def mapOperatingCurrent(self, propertiesDict) -> None:
        nfcCurrent = propertiesDict["NFC3:Current"].find("./Format/Scale").attrib["Default"]
        setPropertyValue(propertiesDict["OTConfig-2:DefaultOpCurrent"].find("./Format/Scale"), nfcCurrent)
        return

    def getNibble(self, value):
        nibble  = (int(value) >> 3) & 0x1
        nibble += (int(value) >> 4) & 0b1110
        return nibble

    def mapOperatingTime(self, propertiesDict) -> None:
        timeRaw = int(propertiesDict["NFC3:OpsTime"].find("./Format/Scale").attrib["Default"])
        time  = self.getNibble(timeRaw) & 0xFF
        time += (self.getNibble(timeRaw >>  8) & 0xFF) <<  4
        time += (self.getNibble(timeRaw >> 16) & 0xFF) <<  8
        time += (self.getNibble(timeRaw >> 24) & 0xFF) << 12
        time *= 4 * 60      # convert from 4hours values to minutes
        setPropertyValue(propertiesDict["Info-0:LampOperationCounter"].find("./Format/Scale"),time)
        return

    def mapClo(self, propertiesDict) -> None:
        cloTimes = [0] * 8
        cloLevels = [0] * 8
        levelPart = int(propertiesDict["NFC3:CLO-Adjust"].find("./Format/Scale").attrib["Default"])
        for i in range (8):
            nfc3clo = int(propertiesDict[f"NFC3:CLO{i+1}"].find("./Format/Scale").attrib["Default"])
            cloTimes[i] = nfc3clo & 0xF
            cloLevels[i] = nfc3clo >> 4
            cloLevels[i] += ((levelPart >> i*2) & 0x3) << 4

        isCloEnabled = self.IsCloEnabled(cloLevels)
        # convert collected clo values to real values
        for i in range (8):
            cloTimes[i] = round((cloTimes[i] * 8192) / 1000)
            cloLevels[i] = int((((cloLevels[i] + 64) * 100) / 128) + 1)

        # detect padding for unsused clo entries
        lastIndex = 7
        last = propertiesDict["NFC3:CLO8"].find("./Format/Scale").attrib["Default"]
        for i in range(7, 0, -1):
            cur = propertiesDict[f"NFC3:CLO{i}"].find("./Format/Scale").attrib["Default"]
            if cur != last:
                lastIndex = i
                break

        setPropertyValue(propertiesDict["ConstLum-0:Command-ClmEnable"].find("./Format/Scale"), isCloEnabled)
        for i in range (8):
            scaleTime = propertiesDict[f"ConstLum-0:Time{i+1}"].find("./Format/Scale")
            scaleLevel = propertiesDict[f"ConstLum-0:AdjustmentLevel{i+1}"].find("./Format/Scale")
            if i <= lastIndex:
                setPropertyValue(scaleTime, cloTimes[i])
                setPropertyValue(scaleLevel, cloLevels[i])
            else:
                setPropertyValue(scaleTime, scaleTime.attrib["Off"])
                setPropertyValue(scaleLevel, scaleTime.attrib["Off"])
        return

    def IsCloEnabled(self, levels):
        for i in range (8):
            if (levels[i] < 0x3F):
                return "1"
        return "0"

    def mapProperties(self, properties) -> None:
        propertiesDict = {}
        for prop in properties:
            propertiesDict[prop.attrib["PgmName"]] = prop
        self.mapOperatingCurrent(propertiesDict)
        self.mapOperatingTime(propertiesDict)
        self.mapClo(propertiesDict)
        return

class DaliHandler(RawHandler):
    def __init__(self, data, allocInterface = "DALI"):
        super().__init__(data)
        self.interfacetype=allocInterface.lower() 
        self.allocInterface = allocInterface   

        instances = 0
        datas = self.data.findall(".//Data")
        for data in datas:
            tmp = int(data.attrib["Instance"])
            if tmp > instances:
                instances = tmp
        self.instances = instances + 1

        self.rawbuffer = [None] * (instances + 1)
        for instance in range(instances + 1):  

            buffer = {}
            banks = self.data.findall(".//Data[@Bank]")
            for bank in banks:
                if bank.attrib["Instance"] == str(instance): 
                    buffer[int(bank.attrib["Bank"])] = bytes.fromhex(bank.attrib["Data"])

            self.rawbuffer[instance] = buffer 

        logger.debug("Raw data has %d instance(s)"%(instances+1))

        return


    def getidentification(self) -> tuple:
        data = self.data.find(".//Data[@Bank='0']")
        if data != None:

            rawbank0 = bytes.fromhex(data.attrib["Data"])
            #print(rawbank0)
            gtin = int.from_bytes(rawbank0[3:9], "big")   # last index = start index + length
            fw   = int.from_bytes(rawbank0[9:10], "big")   
            hw   = int.from_bytes(rawbank0[19:20], "big")  

            logger.debug("Raw file has: %i %i %i"%(gtin, fw, hw))
        else:
            logger.error("no DALI identification data in file")
            sys.exit(-1)

        return (gtin, fw, hw)


    def getddfile(self, baseurl, apikey, driverid) -> et._ElementTree:
        return super().getddfile(baseurl, apikey, gtin=driverid[0], fw=driverid[1], hw=driverid[2])


    def getRaw4Property(self, prop, instance=0, signed=False) -> bytes:
        
        pname = prop.attrib["PgmName"]
        # if pname=="alldata:float":
        #     print("start debug")

        allocs = prop.findall(".//Allocation")

        for alloc in allocs:
            if "Interface" not in alloc.attrib or alloc.attrib["Interface"]==self.allocInterface:
                bank      = int(alloc.find("MemAddress").attrib["Bank"])
                offset    = int(alloc.find("MemAddress").attrib["Byte"])
                bytelen   = int(alloc.find("Size").attrib["ByteCount"])
                try:
                    bitoffset = int(alloc.find("MemAddress").attrib["Bit"])
                except:
                    bitoffset = 0
                # bitoffset = int(alloc.find("MemAddress").attrib["Bit"])
                bitlen    = int(alloc.find("Size").attrib["BitCount"])

                if "NetworkByteOrder" in alloc.attrib:
                    tmp = to_bool(alloc.attrib["NetworkByteOrder"]) 
                    byteorder = "big" if tmp==True else "little"
                else: 
                    byteorder = "big"   # default case, when no byteorder is given in the source PF file  
                
                if bitlen == 0:
                    bytemode = True
                else:
                    bytemode = False

                    if bitlen >8:
                        print("error: bitwise property handling larger than one byte is not supported in this version.")
                        sys.exit()  

                    if bytelen>0:
                        print("error in datamodel: mixed bit and byte mode not supported")
                        sys.exit() 

                if bank not in self.rawbuffer[instance]:
                    print("Missing Memory Bank %s in raw data!"%bank)
                    return None
                if offset+bytelen > len(self.rawbuffer[instance][bank]):
                    print("### warning: raw data is not sufficient. Using default for property %s"%pname)
                    return None
                else:
                    if bytemode:
                        # if bank==207:
                        #     print("*", len(self.rawbuffer[instance][bank]))
                        rawdata = self.rawbuffer[instance][bank][offset:offset+bytelen] 
                        if bytelen<=8:
                            rawstr = str(int.from_bytes(rawdata, byteorder, signed=signed))
                        else:
                            rawstr = rawdata.hex()
                            #print("###:", rawstr , type(rawstr))
                    else:  # bitmode
                        rawint = self.rawbuffer[instance][bank][offset] >> bitoffset       # implicit conversion to int 
                        mask = (2 ** bitlen) - 1
                        rawint = rawint & mask
                        if signed==True: 
                            rawint = subbyte_to_int(rawint, bitlen)
                        rawstr = str(rawint)
                    return rawstr

        return None


    def copyDaliProperties(self, target, instance): 
        daliprops = target.findall(".//DALIProperties/Property") 
        for daliprop in daliprops:
            name = daliprop.attrib["PgmName"]        
            # if name=="RGBWAFPowerOn":
            #     print("start debug")
            propsource = self.data.find(".//Data[@Property='%s'][@Instance='%s']"%(name,str(instance)))
            if propsource != None:
                raw = int(propsource.attrib["Data"], base=16)

                format = daliprop.find(".//Format")
                for sub in format:  
                    use_this_format_element = False
                    if "Instance" not in sub.attrib:
                        if instance==0:
                            use_this_format_element = True 
                    elif int(sub.attrib["Instance"])==instance:
                        use_this_format_element = True

                    if use_this_format_element==True:       # get the datatype for this property from dd file                          
                        datatype = None
                        if sub.tag=="Scale" or sub.tag=="EnumValues" or sub.tag=="ExpValues":
                            datatype = sub.attrib["Type"]
                        if sub.tag=="HexBytes":
                            datatype = "HexBytes"
                            raw = propsource.attrib["Data"]
                            missing0s = int(sub.attrib["Length"]) * 2 - len(raw)
                            if missing0s > 0:
                                raw = ("0" * missing0s) + raw
                        if datatype == None:
                            print("error 3, no valid data format in property %s"%daliprop.attrib["PgmName"])
                        setPropertyValue(sub, raw)
        return 

    def mapProperties(self, properties) -> None:
        # special handling of bio page and encrypted pwds in case of dali
        # They dont have a dali allocation so cannot be read from the raw.
        # But in case of Dali the 'normal' pwds are encrypted so no need for the explicit encrypted 
        # pwd locations (needed in NFC)
        # As a solution we just copy the passwords from the normal one to the encrypted.
        propertiesDict = {}
        for prop in properties:
            propertiesDict[prop.attrib["PgmName"]] = prop
        # handle BIO page
        if 'BIO-0:MasterPwd' in propertiesDict:
            pwd = int(propertiesDict["MSK-0:MasterPwd"].find("./Format/Scale").attrib["Default"])
            setPropertyValue(propertiesDict["BIO-0:MasterPwd"].find("./Format/Scale"), pwd)
        if 'BIO-0:ServicePwd' in propertiesDict:
            pwd = int(propertiesDict["MSK-0:ServicePwd"].find("./Format/Scale").attrib["Default"])
            setPropertyValue(propertiesDict["BIO-0:ServicePwd"].find("./Format/Scale"), pwd)
        if 'BIO-0:PermUser' in propertiesDict:
            pwd = int(propertiesDict["MSK-0:PermUser"].find("./Format/Scale").attrib["Default"])
            setPropertyValue(propertiesDict["BIO-0:PermUser"].find("./Format/Scale"), pwd)
        if 'BIO-0:PermService' in propertiesDict:
            pwd = int(propertiesDict["MSK-0:PermService"].find("./Format/Scale").attrib["Default"])
            setPropertyValue(propertiesDict["BIO-0:PermService"].find("./Format/Scale"), pwd)
        # handle OEM key bio page equivalent
        if 'Pwd1-1:EncryptedPassword' in propertiesDict:
            pwd = int(propertiesDict["Pwd1-1:Password"].find("./Format/Scale").attrib["Default"])
            setPropertyValue(propertiesDict["Pwd1-1:EncryptedPassword"].find("./Format/Scale"), pwd)
        if 'Pwd2-1:EncryptedPassword' in propertiesDict:
            pwd = int(propertiesDict["Pwd2-1:Password"].find("./Format/Scale").attrib["Default"])
            setPropertyValue(propertiesDict["Pwd2-1:EncryptedPassword"].find("./Format/Scale"), pwd)
        return

class Inv2Handler(RawHandler):
    def __init__(self, data, allocInterface = "INV2"):
        super().__init__(data)
        self.interfacetype=allocInterface.lower() 
        self.allocInterface = allocInterface   

        instances = 0
        datas = self.data.findall(".//Data")
        for data in datas:
            tmp = self.getInstanceValue(data)
            if tmp > instances:
                instances = tmp
        self.instances = instances + 1

        self.rawbuffer = [None] * (instances + 1)
        for instance in range(instances + 1):  

            buffer = {}
            banks = self.data.findall(".//Data[@Bank]")
            for bank in banks:
                bankInstance = self.getInstanceValue(bank)
                if bankInstance == instance: 
                    buffer[int(bank.attrib["Bank"])] = bytes.fromhex(bank.attrib["Data"])

            self.rawbuffer[instance] = buffer 

        logger.debug("Raw data has %d instance(s)"%(instances+1))

        return

    def getInstanceValue(self, xmlElem):
        if "Instance" in xmlElem.attrib:
            return int(xmlElem.attrib["Instance"])
        return 0

    def getidentification(self) -> tuple:
        data = self.data.find(".//Header")
        return (data.attrib["Gtin"], data.attrib["Fw"], data.attrib["Hw"])


    def getddfile(self, baseurl, apikey, driverid) -> et._ElementTree:
        return super().getddfile(baseurl, apikey, gtin=driverid[0], fw=driverid[1], hw=driverid[2])


    def getRaw4Property(self, prop, instance=0, signed=False) -> bytes:
        
        pname = prop.attrib["PgmName"]
        # if pname=="alldata:float":
        #     print("start debug")

        allocs = prop.findall(".//Allocation")

        for alloc in allocs:
            if "Interface" not in alloc.attrib or alloc.attrib["Interface"]==self.allocInterface:
                bank      = int(alloc.find("MemAddress").attrib["Bank"])
                offset    = int(alloc.find("MemAddress").attrib["Byte"])
                bytelen   = int(alloc.find("Size").attrib["ByteCount"])
                try:
                    bitoffset = int(alloc.find("MemAddress").attrib["Bit"])
                except:
                    bitoffset = 0
                # bitoffset = int(alloc.find("MemAddress").attrib["Bit"])
                bitlen    = int(alloc.find("Size").attrib["BitCount"])

                if "NetworkByteOrder" in alloc.attrib:
                    tmp = to_bool(alloc.attrib["NetworkByteOrder"]) 
                    byteorder = "big" if tmp==True else "little"
                else: 
                    byteorder = "big"   # default case, when no byteorder is given in the source PF file  
                
                if bitlen == 0:
                    bytemode = True
                else:
                    bytemode = False

                    if bitlen >8:
                        print("error: bitwise property handling larger than one byte is not supported in this version.")
                        sys.exit()  

                    if bytelen>0:
                        print("error in datamodel: mixed bit and byte mode not supported")
                        sys.exit() 

                if bank not in self.rawbuffer[instance]:
                    print("Missing Memory Bank %s in raw data!"%bank)
                    return None
                if offset+bytelen > len(self.rawbuffer[instance][bank]):
                    print("### warning: raw data is not sufficient. Using default for property %s"%pname)
                    return None
                else:
                    if bytemode:
                        # if bank==207:
                        #     print("*", len(self.rawbuffer[instance][bank]))
                        rawdata = self.rawbuffer[instance][bank][offset:offset+bytelen] 
                        if bytelen<=8:
                            rawstr = str(int.from_bytes(rawdata, byteorder, signed=signed))
                        else:
                            rawstr = rawdata.hex()
                            #print("###:", rawstr , type(rawstr))
                    else:  # bitmode
                        rawint = self.rawbuffer[instance][bank][offset] >> bitoffset       # implicit conversion to int 
                        mask = (2 ** bitlen) - 1
                        rawint = rawint & mask
                        if signed==True: 
                            rawint = subbyte_to_int(rawint, bitlen)
                        rawstr = str(rawint)
                    return rawstr

        return None


    def copyDaliProperties(self, target, instance): 
        return 

# endregion
# region ###################  aux function  ###################################


sensortable = [
                {"Rs":0,   "T0":25, "B":3455, "R0":10000, "name":"NCP18XH103J"},
                {"Rs":0,   "T0":25, "B":3998, "R0":15000, "name":"NCP18XW153J" },
                {"Rs":390, "T0":25, "B":3998, "R0":15000, "name":"NCP15XW153E+390"},
                {"Rs":0,   "T0":25, "B":4000, "R0":47000, "name":"EPCOS B57423V2473H"},
                {"Rs":2000, "T0":25, "B":3998, "R0":15000, "name":"NCP15XW153E03RC+2000"},
            ]

def calc_ntc_value(selectedsensor: int, temperature: int):
    s = sensortable[selectedsensor]
    val = (s["R0"] * math.exp(s["B"] * (1 / (temperature + 273.15) - 1 / (s["T0"] + 273.15)))) + s["Rs"]
    return round(val)

def calc_temp_from_ntc(sensortype, resistance):
    s = sensortable[sensortype]
    temperature = ((s["B"] * (s["T0"] + 273.15)) / (((s["T0"] + 273.15) * math.log((resistance - s["Rs"]) / s["R0"])) + s["B"])) - 273.15
    return round(temperature)

def guesssensortype (t25, t40, t55, t70, t85, t100):
    sensortype = -1
    for index, sensor in enumerate(sensortable):
        val25  = calc_ntc_value(index, 25)
        val40  = calc_ntc_value(index, 40)
        val55  = calc_ntc_value(index, 55)
        val70  = calc_ntc_value(index, 70)
        val85  = calc_ntc_value(index, 85)
        val100 = calc_ntc_value(index, 100)

        if t25==val25 and t40==val40 and t55==val55 and t70==val70 and t85==val85 and t100==val100:
            sensortype = index
            break            

    return sensortype

def handle_tpm(structure):
    enabled = structure.find(".//Property[@PgmName='ThermProt-2:Command-TempMeasureEnable']/Format/Scale").attrib["Default"]
    if enabled == "1":
        t25 = int(structure.find(".//Property[@PgmName='ThermProt-2:SensorLevel25']/Format/Scale").attrib["Default"])
        t40 = int(structure.find(".//Property[@PgmName='ThermProt-2:SensorLevel40']/Format/Scale").attrib["Default"])
        t55 = int(structure.find(".//Property[@PgmName='ThermProt-2:SensorLevel55']/Format/Scale").attrib["Default"])
        t70 = int(structure.find(".//Property[@PgmName='ThermProt-2:SensorLevel70']/Format/Scale").attrib["Default"])
        t85 = int(structure.find(".//Property[@PgmName='ThermProt-2:SensorLevel85']/Format/Scale").attrib["Default"])
        t100 = int(structure.find(".//Property[@PgmName='ThermProt-2:SensorLevel100']/Format/Scale").attrib["Default"])
        sensor = guesssensortype(t25, t40, t55, t70, t85, t100)
        sensortarget = structure.find(".//Property[@PgmName='SensorType']/Format/Scale")
        setPropertyValue(sensortarget, sensor)


        if sensor != -1:   # error: no matching sensor found
            rstart = int(structure.find(".//Property[@PgmName='ThermProt-2:StartDeratingResistance']/Format/Scale").attrib["Default"])
            tstart = calc_temp_from_ntc(sensor, rstart)
            starttarget = structure.find(".//Property[@PgmName='PowerReductionStartTemperature']/Format/Scale")
            setPropertyValue(starttarget, tstart)

            rend = int(structure.find(".//Property[@PgmName='ThermProt-2:EndDeratingResistance']/Format/Scale").attrib["Default"])
            tend = calc_temp_from_ntc(sensor, rend)
            endtarget = structure.find(".//Property[@PgmName='PowerReductionEndTemperature']/Format/Scale")
            setPropertyValue(endtarget, tend)

            roff = int(structure.find(".//Property[@PgmName='ThermProt-2:ShutOffResistance']/Format/Scale").attrib["Default"])
            shutdown = int(structure.find(".//Property[@PgmName='ThermProt-2:ShutOffResistance']/Format/Scale").attrib["Off"])
            if roff!=shutdown:
                toff = calc_temp_from_ntc(sensor, roff)
                offtarget = structure.find(".//Property[@PgmName='ShutDownTemperature']/Format/Scale")
                setPropertyValue(offtarget, toff)   
            else:
                offtarget = structure.find(".//Property[@PgmName='ShutDownTemperature']/Format/Scale")
                setPropertyValue(offtarget, shutdown)
    return


def to_bool(value):    # accepts multiple types
    if type(value) == type(''):   # string like types
        if value.lower() in ("yes", "y", "true",  "t", "1"):
            return True
        if value.lower() in ("no",  "n", "false", "f", "0", ""):
            return False
        raise Exception('Invalid value for boolean conversion: ' + value)
    return bool(value)


def get_structure_from_xml_file(fname):

    parser = et.XMLParser(remove_blank_text=True)
    if not os.path.exists(fname):
        print("File '%s' not found!"%fname)
        sys.exit(-1)
    try:
        data = et.parse(fname, parser)
    except et.XMLSyntaxError:
        num=len(parser.error_log)
        print("%d errors in %s found"%(num, fname))
        for i in range (num):
            error = parser.error_log[i]
            print("* ",error.message, " @line", error.line,"/column", error.column)
        sys.exit(0)
    return data


def subbyte_to_int (value: int, bitlen: int) -> int:
    sign = 1 << (bitlen-1)
    if value & sign != 0:    # negative value: extent sign to leading bits
        for i in range(bitlen, 8):
            sign = sign << 1 
            value = value | sign 
        return (int.from_bytes(value.to_bytes(1,'big'), byteorder='big', signed=True))
    else:                   # no special handling for positive values
        return value

    ## bitwise ops in py are defined on int only. alternative for bitwise ops on bytearrays:
    ## bytearray(map (lambda a, b: a&b, x , y))     where x,y are the param of type bytearray


def set_xml_to_default(structure):
    daliprops = structure.findall(".//DALIProperties/Property")  
    if daliprops != None:
        for props in daliprops:
            for para in props.iter():
                if "Default" in para.attrib:
                    para.attrib["Default"] = para.attrib["Default"]   

    props = structure.findall(".//SubDevices/SubDevice/Property")
    for prop in props:
        pname = prop.attrib["PgmName"]
        if pname.find(":")>=0:                  # avoid copying device constants, only copy the Default for parameters
            for para in prop.iter():
            #tmp = para.find("..").find("..")                
            # if "PgmName" in tmp.attrib:
            #     pname = tmp.attrib["PgmName"]
            #     if pname.find(":")>=0:            
                if "Default" in para.attrib:
                    para.attrib["Default"] = para.attrib["Default"]   
                if para.tag == "Scale":
                    if "Offset" in para.attrib:
                        to = para.attrib["Offset"]
                    else:
                        to = "0.0"
                    if "Multiplier" in para.attrib:
                        tm = para.attrib["Multiplier"]
                    else:
                        tm = "1.0"
                    para.attrib["Offset"] = to
                    para.attrib["Multiplier"] = tm

    return structure     




# endregion
# region ###################  interface functions  ############################

def setProgrammingTrue(ref):
    if ref is None:
        return
    if "Programming" in ref.attrib:
        if "LockProgramming" in ref.attrib:
            if ref.attrib["LockProgramming"]=="false":   # only if change is allowed
                ref.attrib["Programming"]="true" 
        else:
            ref.attrib["Programming"]="true" 
    return

def handleSpecialCaseDaliAddress(desttree):
    values = desttree.findall(".//Property[@PgmName='Address']/Format/Scale")
    if values != None:
        for value in values:
            if "Default" in value.attrib:
                # programming with 'unassign' will write 255 into the tag, readback for unassigned address is 255
                # and needs to be translated into unassign, in case of Dali pi its done in the P4
                if value.attrib["Default"] == "255":
                    value.attrib["Default"] = "254"
                # programming with 'unassign' will write 128 into the tag, the FW detects this value as illegal and does nothing
                # if the driver was not power cycled, the readback contains that illegal value and need to be translated into keep.
                if value.attrib["Default"] == "128":
                    value.attrib["Default"] = "255"


def convert_raw_2_project(xmltree, baseurl="https://www.tuner4tronic.com", apikey=""):

    raw = xmltree.find("//Interface")           # just use the first driver in the file
                                                # and ignore the rest (if any)

    ### Step 1: which type of interface is present?  
    interfacetype = raw.attrib["Type"].lower()
    logger.debug("Raw File has Interfacetype %s"%interfacetype)

    if interfacetype=="dali":
        handler = DaliHandler(raw)
    elif interfacetype=="nfc1":
        handler = Nfc1Handler(raw)
    elif interfacetype=="nfc2":
        handler = Nfc2Handler(raw)
    elif interfacetype=="nfc3":
        handler = Nfc3Handler(raw)
    elif interfacetype=="osrser":
        handler = OsrHandler(raw)
    elif interfacetype=="inv2":
        handler = Inv2Handler(raw)
    elif interfacetype=="bt":
        handler = DaliHandler(raw, "BT")
    else:
        logger.error("unsupported interface type %s in source file"%interfacetype)
        sys.exit(0)

    ### Step 2: identify the driver  
    deviceIdentification = handler.getidentification()
    logger.debug("Found driver %s"%str(deviceIdentification))

    ### Step 3: get the dd file 
    if deviceIdentification[0]==18838586676582:
        # for testing: using a static dd file for that gtin   
        desttree = get_structure_from_xml_file("testdata.xml")     
    else:
        try:
            desttree = handler.getddfile(baseurl, apikey, deviceIdentification)
        except:
            raise err.InputError("got unknown driver with GTIN %s and FW %s"%(deviceIdentification[0], deviceIdentification[1]))
        scope = desttree.find(".//Scope").text
        if "C" not in scope and "F" not in scope:
            raise err.InputError("got unsupported driver with GTIN %s and FW %s"%(deviceIdentification[0], deviceIdentification[1]))

    # gtin = desttree.find(".//GTIN").text
    # fw = desttree.find(".//FW_Major").text
    # hw = desttree.find(".//HW_Major").text

    # logger.debug("DD file has: %s, %s, %s"%(gtin, fw, hw))

    # mark the xml as raw, needed in converter to handle keys
    desttree.find(".//SubDevices").attrib["IsRaw"] = "1"
    prop_is_signed = [
        'AstroA-0:DimStartTime',
        'AstroA-0:Latitude',
        'AstroA-0:Longitude',
        'AstroA-0:UTCTimeShift',
        '3DIM:DimShift_1',
        '3DIM:DimShift_2',
        '3DIM:Latitude',
        '3DIM:Longitude',
        '3DIM:TimeShiftToGMT',
        'Astro-0:DimShift_1',
        'Astro-0:DimShift_2',
        'Astro-0:Latitude',
        'Astro-0:Longitude',
        'Astro-0:TimeShiftToGMT',
        'Astro-1:DimStartTime',
        'Astro-1:Latitude',
        'Astro-1:Longitude',
        'Astro-1:UTCTimeShift',
        'Astro-2:DimStartTime',
        'Astro-2:Latitude',
        'Astro-2:Longitude',
        'Astro-2:UTCTimeShift',
        'Astro-3:DimStartTime',
        'Astro-3:Latitude',
        'Astro-3:Longitude',
        'Astro-3:UTCTimeShift',
        'DG-0:PrestartDerating',
        'TWA-0:Ta',
        'TWA-0:Tmod_Ch0',
        'TWA-0:Tmod_Ch1',
        'alldata:sig',      # dummy for testcase        
        'alldata:sigbits',  # dummy for testcase
        ]   

    for instance in range(handler.getInstances()):
        logger.debug("Property collecting: starting with instance %s"%instance)

        handler.copyDaliProperties(desttree, instance)

        subdevs = desttree.findall(".//SubDevice")
        for subdev in subdevs:

            multi = False
            if "MultipleInstance" in subdev.attrib:
                multi = to_bool(subdev.attrib["MultipleInstance"])

            if instance > 0 and multi == False:     # despite multiple instances in the raw file, this SubDevice has only one, 
                continue                            # continue with next subdevice

            props = subdev.findall(".//Property")
            for prop in props:

                pname = prop.attrib["PgmName"]
                if pname.find(":")>=0 and not (pname=="TWA-0:Voltage1" or pname=="TWA-0:Voltage2"):  
                                                                    # fix error of old FW1 dd files                
                    access = prop.attrib["AccessType"]  # do not copy "write only" properties
                    if not (access == "W" or access=="WL"):

                        # if pname =="DG-0:PrestartDerating":
                        #     print("ping")     # set breakpoint here...

                        if pname in prop_is_signed:
                            signed = True
                        else:
                            signed = False

                        tmp = handler.getRaw4Property(prop, instance=instance, signed=signed)

                        if tmp != None:    # only set existing values
                            handler.setValuefromRaw(tmp, prop, instance=instance)
                        else:
                            logger.debug("Skipping Property %s which does not exist in raw."%pname)

    properties = desttree.findall(".//Property")
    handler.mapProperties(properties)

    prog = desttree.find(".//OperatingModes")
    setProgrammingTrue(prog)
    prog = desttree.find(".//DALIProperties")
    setProgrammingTrue(prog)
    progs = desttree.findall(".//SubDevice")
    for prog in progs:
        setProgrammingTrue(prog) 

    # finally handle the special case(s) for "raw to project": 
    # when driver has TPM feature  with sensor-based mode, "detect" the sensor type from settings 
    # and bring it into SensorType as Default
    tpm = desttree.find(".//SubDevice[@PgmName='ThermProt-2']") 
    if tpm != None: 
        handle_tpm(tpm)

    # handle special Dali Address value, in case of programming Address over NFC,
    # T4T-P uses a special value (128) as 'do not change address' value. It must be 
    # converted back to a valid value
    handleSpecialCaseDaliAddress(desttree)

    descr = desttree.find(".//MetaData/DeviceDescription")
    descr.text = "Project imported from driver data readout V " + toolversion

    return desttree


# endregion
# region ###################  main = module tests  ############################


if __name__ == "__main__":

    # sourcefilename = "driverdata.xml"   # static example
    # sourcestructure = get_structure_from_xml_file (sourcefilename)
    # result = convert_raw_2_project(sourcestructure) 

    # result.write(open("dalioutput.xml", 'bw'),
    #         pretty_print=True,
    #         xml_declaration=True,
    #         encoding='utf-8') 

    sourcefilename = r"C:\Users\BertWeber\Downloads\12028-driverdata.raw"   # static example
    sourcestructure = get_structure_from_xml_file (sourcefilename)
    result = convert_raw_2_project(sourcestructure, "https://staging.tuner4tronic.com", "osramalpha") 

    result.write(open("nfcoutput.xml", 'bw'),
            pretty_print=True,
            xml_declaration=True,
            encoding='utf-8') 


    # ## test the sensor value calulation
    # for index in range (4):
    #     print(calc_ntc_value(index, 25), 
    #         calc_ntc_value(index, 40),
    #         calc_ntc_value(index, 55),
    #         calc_ntc_value(index, 70),
    #         calc_ntc_value(index, 85),
    #         calc_ntc_value(index, 100))

    # print(calc_temp_from_ntc(1,15000))
    # print(calc_temp_from_ntc(1,7891))
    # print(calc_temp_from_ntc(1,4402))
    # print(calc_temp_from_ntc(1,2585))
    # print(calc_temp_from_ntc(1,1587))
    # print(calc_temp_from_ntc(1,1013))


# endregion ########### main ######################
